<?php
/**
 * Script para agregar emails a los archivos de datos de carreras
 * Toma los datos de data_3k.json, data_5k.json y data_10k.json
 * y les agrega el campo "Email" basándose en cronoh_emails.csv
 */

// Función para leer el archivo CSV y crear un array asociativo
function loadEmailsFromCSV($csvFile) {
    $emails = [];
    
    if (!file_exists($csvFile)) {
        echo "Error: No se encontró el archivo CSV: $csvFile\n";
        return $emails;
    }
    
    $handle = fopen($csvFile, 'r');
    if ($handle === false) {
        echo "Error: No se pudo abrir el archivo CSV: $csvFile\n";
        return $emails;
    }
    
    // Leer la primera línea (encabezados)
    $headers = fgetcsv($handle);
    
    // Leer cada línea del CSV
    while (($data = fgetcsv($handle)) !== false) {
        $numeroCorredor = trim($data[0]);
        $correoCorredor = trim($data[1]);
        
        // Solo agregar si hay un número de corredor válido
        if (!empty($numeroCorredor) && is_numeric($numeroCorredor)) {
            // Si el email está vacío o es null@null.com, usar un email por defecto
            if (empty($correoCorredor) || $correoCorredor === 'null@null.com') {
                $emails[$numeroCorredor] = '';
            } else {
                $emails[$numeroCorredor] = $correoCorredor;
            }
        }
    }
    
    fclose($handle);
    echo "Cargados " . count($emails) . " emails del archivo CSV.\n";
    return $emails;
}

// Función para procesar un archivo JSON y agregar emails
function processJsonFile($jsonFile, $emails) {
    if (!file_exists($jsonFile)) {
        echo "Error: No se encontró el archivo JSON: $jsonFile\n";
        return false;
    }
    
    $jsonContent = file_get_contents($jsonFile);
    $data = json_decode($jsonContent, true);
    
    if ($data === null) {
        echo "Error: No se pudo decodificar el JSON del archivo: $jsonFile\n";
        return false;
    }
    
    $updatedCount = 0;
    
    // Procesar cada corredor
    foreach ($data as &$runner) {
        // Verificar que el runner tenga la estructura correcta
        if (isset($runner['Bib']) && !empty($runner['Bib'])) {
            $bib = $runner['Bib'];
            
            // Buscar el email correspondiente
            if (isset($emails[$bib])) {
                $runner['Email'] = $emails[$bib];
                $updatedCount++;
            } else {
                // Si no se encuentra el email, agregar campo vacío
                $runner['Email'] = '';
            }
        }
    }
    
    // Crear el nombre del archivo de salida
    $pathInfo = pathinfo($jsonFile);
    $outputFile = $pathInfo['dirname'] . '/' . $pathInfo['filename'] . '_with_email.json';
    
    // Guardar el archivo actualizado
    $jsonOutput = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    if (file_put_contents($outputFile, $jsonOutput) !== false) {
        echo "Archivo procesado: $jsonFile\n";
        echo "Archivo de salida: $outputFile\n";
        echo "Corredores actualizados: $updatedCount\n";
        echo "---\n";
        return true;
    } else {
        echo "Error: No se pudo guardar el archivo: $outputFile\n";
        return false;
    }
}

// Ejecutar el script
echo "=== Script para agregar emails a datos de carreras ===\n\n";

// Rutas de los archivos
$csvFile = '../data/cronoh_emails.csv';
$jsonFiles = [
    '../data/data_3k.json',
    '../data/data_5k.json',
    '../data/data_10k.json',
    '../data/data_21k.json'
];

// Cargar emails del CSV
echo "Cargando emails del archivo CSV...\n";
$emails = loadEmailsFromCSV($csvFile);

if (empty($emails)) {
    echo "No se pudieron cargar los emails. Terminando el script.\n";
    exit(1);
}

// Procesar cada archivo JSON
echo "\nProcesando archivos JSON...\n";
$totalProcessed = 0;

foreach ($jsonFiles as $jsonFile) {
    if (processJsonFile($jsonFile, $emails)) {
        $totalProcessed++;
    }
}

echo "\n=== Resumen ===\n";
echo "Archivos procesados exitosamente: $totalProcessed de " . count($jsonFiles) . "\n";
echo "Script completado.\n";

?>
