// Importar la configuración
import CONFIG from '../../config.js';
import { assignPhotosToModal } from './photo-timing.js';

// Función de debounce para mejorar el rendimiento
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Scripts para la búsqueda y filtrado de resultados
document.addEventListener('DOMContentLoaded', function() {
    const ITEMS_PER_PAGE = 10;
    let currentPage = 1;
    let totalPages = 0;
    
    // Esperar a que los datos se carguen
    document.addEventListener('resultsDataLoaded', function() {
        updateCategorySelect();
        filterResults();
    });
    
    // Función para actualizar el select de categorías con los datos cargados
    function updateCategorySelect() {
        const categorySelect = document.getElementById('filter-category');
        const currentValue = categorySelect.value; // Preservar la selección actual
        
        // Obtener categorías únicas de los datos
        const categories = [...new Set(resultsData.map(runner => runner.category))].filter(Boolean).sort();
        
        // Limpiar opciones existentes excepto "Todas las categorías"
        categorySelect.innerHTML = '<option value="">Todas las categorías</option>';
        
        // Agregar las categorías encontradas en los datos
        categories.forEach(category => {
            const option = document.createElement('option');
            option.value = category;
            option.textContent = category;
            categorySelect.appendChild(option);
        });
        
        // Restaurar la selección anterior si aún existe
        if (currentValue && categories.includes(currentValue)) {
            categorySelect.value = currentValue;
        }
        
        console.log('Categorías encontradas:', categories);
    }
    
    // Función para renderizar los resultados en la tabla con paginación
    function renderResults(results) {
        console.log('renderResults llamado con:', results.length, 'registros');
        if (results.length > 0) {
            console.log('Primer resultado:', results[0]);
        }
        
        const tbody = document.querySelector('.results-table tbody');
        const resultsSection = document.querySelector('.results-section');
        const tableHead = document.querySelector('.results-table thead tr');
        
        // Limpiar el contenido de la tabla
        tbody.innerHTML = '';
        
        // Eliminar tarjetas anteriores si existen
        const existingCards = document.querySelectorAll('.results-card');
        existingCards.forEach(card => card.remove());
        
        if (!results || results.length === 0) {
            const row = document.createElement('tr');
            row.innerHTML = '<td colspan="6" class="no-results">No se encontraron resultados que coincidan con tu búsqueda</td>';
            tbody.appendChild(row);
            document.getElementById('results-count').textContent = '0';
            document.getElementById('total-results').textContent = '0';
            // Ocultar paginación
            document.querySelector('.pagination').style.display = 'none';
            return;
        }
        
        // Actualizar las cabeceras de la tabla con los parciales detectados
        tableHead.innerHTML = '';
        tableHead.innerHTML += '<th>Pos.</th>';
        tableHead.innerHTML += '<th>Corredor</th>';
        if (window.parcialesInfo && window.parcialesInfo.keys && window.parcialesInfo.keys.length > 0) {
            // Añadir encabezados para parciales dinámicos
            window.parcialesInfo.keys.forEach((parcialKey, index) => {
                const label = window.parcialesInfo.labels[index] || parcialKey;
                tableHead.innerHTML += `<th>Parcial ${index + 1}<br><small>(${label})</small></th>`;
            });
        }
        // Añadir encabezados para tiempos finales
        tableHead.innerHTML += '<th>Tiempo Chip</th>';
        tableHead.innerHTML += '<th>Tiempo oficial</th>';
        
        // Mostrar paginación
        document.querySelector('.pagination').style.display = 'flex';
        
        // Detectar si hay un filtro de categoría activo
        const categoryFilter = document.getElementById('filter-category').value;
        const isCategoryFiltered = categoryFilter && categoryFilter !== '';
        
        // Si hay filtro de categoría, necesitamos calcular posiciones dentro de toda la categoría
        let categoryPositions = new Map();
        if (isCategoryFiltered) {
            // Obtener TODOS los corredores de esa categoría (sin filtros de nombre/bib)
            const allInCategory = resultsData.filter(r => r.category === categoryFilter);
            // Ordenar por tiempo oficial (gunTime)
            const sortedByTime = [...allInCategory].sort((a, b) => {
                const timeA = a.gunTime?.time || '99:99:99';
                const timeB = b.gunTime?.time || '99:99:99';
                return timeA.localeCompare(timeB);
            });
            // Crear mapa de UniqueId -> posición en categoría
            sortedByTime.forEach((runner, index) => {
                categoryPositions.set(runner.UniqueId, index + 1);
            });
        }
        
        // Calcular índices de inicio y fin para la página actual
        const startIndex = (currentPage - 1) * ITEMS_PER_PAGE;
        const endIndex = Math.min(startIndex + ITEMS_PER_PAGE, results.length);
        
        // Mostrar solo los resultados de la página actual
        for (let i = startIndex; i < endIndex; i++) {
            const runner = results[i];
            
            // Vista de tabla (para escritorio)
            const row = document.createElement('tr');
            
            // Determinar la posición a mostrar y si es top-3
            let displayPosition, isTop3;
            
            if (isCategoryFiltered) {
                // Si hay filtro de categoría, usar la posición real dentro de toda la categoría
                const categoryPosition = categoryPositions.get(runner.UniqueId) || i + 1;
                displayPosition = categoryPosition + '°';
                isTop3 = categoryPosition <= 3 ? 'top-3' : '';
                
                // Agregar clase a la fila si es top 3 en la categoría
                if (categoryPosition <= 3) {
                    row.classList.add('top-3-row');
                }
            } else {
                // Sin filtro de categoría, mostrar posición general
                displayPosition = runner.position ? runner.position + '°' : 'No definido';
                isTop3 = runner.position && runner.position <= 3 ? 'top-3' : '';
                
                // Agregar clase a la fila si es top 3 en la posición general
                if (runner.position && runner.position <= 3) {
                    row.classList.add('top-3-row');
                }
            }
            
            let rowHTML = `
                <td class="position ${isTop3}">${displayPosition}</td>
                <td>
                    <div class="runner-details">
                        <span class="bib-number">${runner.bib || 'No definido'}</span>
                        <div>
                            <div class="runner-name"><a href="#" class="runner-link" data-bib="${runner.bib}" data-uniqueid="${runner.UniqueId || ''}">${runner.name || 'No definido'}</a></div>
                            <div class="runner-category">${runner.category || 'No definido'} - ${runner.distance || 'No definido'}</div>
                            <div class="actions-cell-horizontal" style="margin: 4px 0 0 0;">
                                <button class="action-btn-rect btn-certificado" title="Ver certificado" data-bib="${runner.bib}" data-uniqueid="${runner.UniqueId || ''}"><i class="fas fa-certificate"></i></button>
                                <button class="action-btn-rect btn-metricas" title="Ver métricas" data-bib="${runner.bib}" data-uniqueid="${runner.UniqueId || ''}"><i class="fas fa-chart-bar"></i></button>
                                <button class="action-btn-rect btn-fotos" title="Ver fotos" data-bib="${runner.bib}" data-uniqueid="${runner.UniqueId || ''}"><i class="fas fa-camera"></i></button>
                            </div>
                        </div>
                    </div>
                </td>`;
                
            // Agregar celdas para los parciales (dinámicos)
            if (window.parcialesInfo && window.parcialesInfo.keys && window.parcialesInfo.keys.length > 0) {
                window.parcialesInfo.keys.forEach((parcialKey, index) => {
                    const mpKey = `mp${index + 1}`;
                    if (runner[mpKey] && typeof runner[mpKey] === 'object') {
                        rowHTML += `
                    <td>
                        <div class="time">${runner[mpKey].time || '00:00:00'}</div>
                        <div class="pace">${runner[mpKey].pace || '0.00 km/h'}</div>
                    </td>`;
                    } else {
                        rowHTML += `
                    <td>
                        <div class="time">00:00:00</div>
                        <div class="pace">0.00 km/h</div>
                    </td>`;
                    }
                });
            }
                
            // Añadir tiempos finales
            rowHTML += `
                <td>
                    <div class="time">${runner.finishTime && runner.finishTime.time ? runner.finishTime.time : '00:00:00'}</div>
                    <div class="pace">${runner.finishTime && runner.finishTime.pace ? runner.finishTime.pace : '0.00 km/h'}</div>
                </td>
                <td>
                    <div class="time">${runner.gunTime && runner.gunTime.time ? runner.gunTime.time : '00:00:00'}</div>
                    <div class="pace">${runner.gunTime && runner.gunTime.pace ? runner.gunTime.pace : '0.00 km/h'}</div>
                </td>
            `;
            
            row.innerHTML = rowHTML;
            
            tbody.appendChild(row);
            
            // Vista de tarjeta (para móvil)
            const card = document.createElement('div');
            card.className = 'results-card';
            
            // Usar la misma lógica de posición que en la vista de tabla
            let positionText;
            if (isCategoryFiltered) {
                // Posición real dentro de toda la categoría
                const categoryPosition = categoryPositions.get(runner.UniqueId) || i + 1;
                positionText = categoryPosition + '°';
                if (categoryPosition <= 3) {
                    card.classList.add('top-3-card');
                }
            } else {
                // Posición general
                positionText = runner.position ? runner.position + '°' : 'No definido';
                if (runner.position && runner.position <= 3) {
                    card.classList.add('top-3-card');
                }
            }
            
            // Generar HTML para la tarjeta móvil
            let cardHTML = `
                <div class="card-header">
                    <span class="position-card ${isTop3}">${positionText}</span>
                    <span class="bib-number-card">${runner.bib || 'No definido'}</span>
                </div>
                <div class="runner-info">
                    <div class="runner-name-card"><a href="#" class="runner-link" data-bib="${runner.bib}" data-uniqueid="${runner.UniqueId || ''}">${runner.name || 'No definido'}</a></div>
                    <div class="runner-category-card">${runner.category || 'No definido'} - ${runner.distance || 'No definido'}</div>
                </div>
                <div class="time-info">
                    <div>
                        <div class="time-label-card">Tiempo oficial</div>
                        <div class="time-value-card">${runner.gunTime && runner.gunTime.time ? runner.gunTime.time : '00:00:00'}</div>
                    </div>
                    <div>
                        <div class="time-label-card">Tiempo Chip</div>
                        <div class="time-value-card">${runner.finishTime && runner.finishTime.time ? runner.finishTime.time : '00:00:00'}</div>
                    </div>
                </div>
                <button class="expand-details-btn"><i class="fas fa-chevron-down"></i> Ver parciales</button>
                <div class="details-section">
                    <div class="parcials-card-grid">`;
            
            // Determinar cuántos parciales mostrar
            const parcialCount = window.parcialesInfo && window.parcialesInfo.count ? 
                Math.min(window.parcialesInfo.count, 5) : 5;
            
            // Generar HTML para cada parcial
            for (let i = 1; i <= parcialCount; i++) {
                const mpKey = `mp${i}`;
                // Obtener la etiqueta del parcial si está disponible
                const parcialLabel = window.parcialesInfo && window.parcialesInfo.labels && window.parcialesInfo.labels[i-1] ? 
                    window.parcialesInfo.labels[i-1] : `km ${i*2}`;
                
                // El último parcial ocupa 2 columnas en la cuadrícula si hay menos de 5 parciales
                const spanStyle = (i === parcialCount && parcialCount < 5) ? 
                    'style="grid-column: span 2;"' : '';
                
                const timeValue = runner[mpKey] && runner[mpKey].time ? runner[mpKey].time : '00:00:00';
                const paceValue = runner[mpKey] && runner[mpKey].pace ? runner[mpKey].pace : '0.00 km/h';
                
                cardHTML += `
                        <div class="parcial-box" ${spanStyle}>
                            <div class="parcial-label">Parcial ${i} <small>(${parcialLabel})</small></div>
                            <div class="parcial-time">${timeValue}</div>
                            <div class="parcial-pace">${paceValue}</div>
                        </div>`;
            }
            
            // Completar el HTML de la tarjeta
            cardHTML += `
                    </div>
                </div>
                <div class="card-actions">
                    <button class="card-btn card-btn-certificate btn-certificado" data-bib="${runner.bib}" data-uniqueid="${runner.UniqueId || ''}" aria-label="Ver certificado"><i class="fas fa-certificate"></i> Certificado</button>
                    <button class="card-btn card-btn-metrics btn-metricas" data-bib="${runner.bib}" data-uniqueid="${runner.UniqueId || ''}" aria-label="Ver métricas"><i class="fas fa-chart-bar"></i> Métricas</button>
                    <button class="card-btn card-btn-photos btn-fotos" data-bib="${runner.bib}" data-uniqueid="${runner.UniqueId || ''}" aria-label="Ver fotos"><i class="fas fa-camera"></i> Fotos</button>
                </div>
            `;
            
            card.innerHTML = cardHTML;
            
            // Insertar la tarjeta antes de la paginación
            const pagination = document.querySelector('.pagination');
            resultsSection.insertBefore(card, pagination);
        }
        
        // Actualizar contador de resultados
        document.getElementById('results-count').textContent = Math.min(ITEMS_PER_PAGE, results.length - startIndex);
        document.getElementById('total-results').textContent = results.length;
        
        // Calcular total de páginas
        totalPages = Math.ceil(results.length / ITEMS_PER_PAGE);
        
        // Asegurar que currentPage no supere el total de páginas
        if (currentPage > totalPages) {
            currentPage = 1;
        }
        
        // Actualizar los botones de paginación
        updatePaginationButtons();
    }
    
    // Función para filtrar resultados optimizada
    function filterResults() {
        const filters = {
            name: document.getElementById('search-name').value.toLowerCase().trim(),
            bib: document.getElementById('search-bib').value.trim(),
            category: document.getElementById('filter-category').value,
            distance: document.getElementById('filter-distance').value,
            gender: document.getElementById('filter-gender').value
        };

        // Verificar si hay algún filtro activo (excluyendo distancia)
        const hasActiveFilters = filters.name || filters.bib || filters.category || filters.gender;

        // Si no hay filtros activos, mostrar todos los resultados
        if (!hasActiveFilters) {
            currentFilteredResults = [...resultsData];
        } else {
            currentFilteredResults = resultsData.filter(runner => {
                // Aplicar filtros solo si están presentes (NO filtrar por distancia aquí)
                if (filters.name && !runner.name.toLowerCase().includes(filters.name)) return false;
                if (filters.bib && !runner.bib.includes(filters.bib)) return false;
                if (filters.category && runner.category !== filters.category) return false;
                // NO aplicar filtro de distancia - los datos ya están filtrados por el selector
                if (filters.gender && runner.gender !== filters.gender) return false;
                return true;
            });
        }
        
        // Si hay filtro de categoría activo, ordenar por tiempo oficial para calcular posiciones correctamente
        if (filters.category) {
            currentFilteredResults.sort((a, b) => {
                // Asegurarse de que gunTime existe y es un objeto
                const timeA = (a.gunTime && a.gunTime.time) ? a.gunTime.time : '99:99:99';
                const timeB = (b.gunTime && b.gunTime.time) ? b.gunTime.time : '99:99:99';
                return timeA.localeCompare(timeB);
            });
        }
        
        currentPage = 1;
        renderResults(currentFilteredResults);
    }

    // Evento para botón de búsqueda
    document.getElementById('search-btn').addEventListener('click', filterResults);
    
    // Eventos para campos de búsqueda con debounce
    const debouncedFilter = debounce(filterResults, 300);
    document.getElementById('search-name').addEventListener('input', debouncedFilter);
    document.getElementById('search-bib').addEventListener('input', debouncedFilter);
    
    // Eventos para filtros select (inmediatos)
    document.getElementById('filter-category').addEventListener('change', filterResults);
    document.getElementById('filter-gender').addEventListener('change', filterResults);
    
    // Evento para cambio de distancia
    document.getElementById('filter-distance').addEventListener('change', function() {
        const distance = this.value;
        window.reloadResultsData(distance);
        // Aplicar filtros después de cargar los nuevos datos
        document.addEventListener('resultsDataLoaded', function onDataLoaded() {
            updateCategorySelect(); // Actualizar categorías con los nuevos datos
            filterResults();
            // Remover el listener para evitar duplicados
            document.removeEventListener('resultsDataLoaded', onDataLoaded);
        });
    });
    
    // Botón de reset mejorado
    document.getElementById('reset-btn').addEventListener('click', function() {
        // Limpiar todos los campos de búsqueda excepto distancia
        document.getElementById('search-name').value = '';
        document.getElementById('search-bib').value = '';
        document.getElementById('filter-category').value = ''; // Resetear a "Todas las categorías"
        // NO resetear filter-distance - mantener la distancia seleccionada
        document.getElementById('filter-gender').value = '';
        
        // Resetear los resultados
        currentFilteredResults = [...resultsData];
        currentPage = 1;
        renderResults(currentFilteredResults);
    });
    
    // Evento para ordenar resultados
    document.getElementById('sort-select').addEventListener('change', function() {
        const sortBy = this.value;
        let sortedResults = [...currentFilteredResults];
        
        switch(sortBy) {
            case 'position':
                sortedResults.sort((a, b) => a.position - b.position);
                break;
            case 'name':
                sortedResults.sort((a, b) => a.name.localeCompare(b.name));
                break;
            case 'time':
                sortedResults.sort((a, b) => {
                    const timeA = a.gunTime?.time || '99:99:99';
                    const timeB = b.gunTime?.time || '99:99:99';
                    return timeA.localeCompare(timeB);
                });
                break;
            case 'category':
                sortedResults.sort((a, b) => a.category.localeCompare(b.category));
                break;
        }
        
        currentFilteredResults = sortedResults;
        renderResults(currentFilteredResults);
    });
    
    // Manejo del modal de detalles de corredor
    document.addEventListener('click', function(e) {
        const runnerLink = e.target.closest('.runner-link, .btn-metricas, .card-btn-metrics');
        if (runnerLink) {
            e.preventDefault();
            const bibNumber = runnerLink.dataset.bib;
            const runner = resultsData.find(r => r.bib === bibNumber);
            if (runner) {
                // Rellenar los datos del corredor en el modal
                document.getElementById('modal-runner-name').textContent = runner.name;
                document.getElementById('modal-runner-category').textContent = `${runner.distance} - ${runner.gender === 'M' ? 'Varonil' : 'Femenil'} - ${runner.category}`;
                document.getElementById('modal-distance').textContent = runner.distance;
                document.getElementById('modal-gender').textContent = runner.gender;
                document.getElementById('modal-category').textContent = runner.category;
                document.getElementById('modal-position').textContent = runner.position;
                
                // Mostrar ID único de verificación
                const uniqueIdElement = document.getElementById('modal-unique-id');
                if (uniqueIdElement) {
                    uniqueIdElement.textContent = runner.UniqueId || '----';
                }
                
                // Calcular el total real de corredores para esta distancia
                const runnersForDistance = resultsData.filter(r => r.distance === runner.distance);
                
                const totalRunners = runnersForDistance.length;
                
                document.getElementById('modal-total-runners').textContent = `${totalRunners} corredores`;
                document.getElementById('modal-runners-before').textContent = runner.position - 1;
                document.getElementById('modal-runners-after').textContent = totalRunners - runner.position;
                
                // Actualizar las etiquetas de los parciales si hay información de parciales disponible
                if (window.parcialesInfo && window.parcialesInfo.keys) {
                    const parcialBoxes = document.querySelectorAll('.times-grid.parcials-grid .time-box');
                    const parcialCount = Math.min(window.parcialesInfo.count, 5); // Máximo 5 parciales
                    
                    // Actualizar las etiquetas de los parciales en el modal
                    for (let i = 0; i < parcialCount; i++) {
                        const parcialBox = parcialBoxes[i];
                        if (parcialBox) {
                            const label = window.parcialesInfo.labels[i] || `Parcial ${i+1}`;
                            const timeLabel = parcialBox.querySelector('.time-label');
                            if (timeLabel) {
                                timeLabel.innerHTML = `Parcial ${i+1} <small>(${label})</small>`;
                            }
                        }
                    }
                    
                    // Ocultar los parciales que no existen
                    for (let i = parcialCount; i < 5; i++) {
                        const parcialBox = parcialBoxes[i];
                        if (parcialBox) {
                            parcialBox.style.display = 'none';
                        }
                    }
                }
                
                                // Tiempos - generar parciales dinámicamente
                const parcialesGrid = document.getElementById('modal-parcials-grid');
                parcialesGrid.innerHTML = ''; // Limpiar parciales anteriores
                
                // Función para calcular minutaje por kilómetro (min/km)
                const calculateMinKm = (timeStr, paceStr) => {
                    if (!timeStr || timeStr === "00:00:00") return "00:00:00 min/km";
                    
                    // Extraer km/h del pace
                    const kmhMatch = paceStr.match(/(\d+(?:\.\d+)?)/);
                    if (kmhMatch) {
                        const kmh = parseFloat(kmhMatch[1]);
                        if (kmh > 0) {
                            const minPerKm = 60 / kmh;
                            const minutes = Math.floor(minPerKm);
                            const seconds = Math.round((minPerKm - minutes) * 60);
                            return `00:${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')} min/km`;
                        }
                    }
                    return "00:00:00 min/km";
                };
                
                // Lista de parciales disponibles en el runner
                const parcialesDisponibles = [];
                for (let i = 1; i <= 5; i++) {
                    const parcialKey = `mp${i}`;
                    if (runner[parcialKey] && runner[parcialKey].time && runner[parcialKey].time !== "00:00:00") {
                        parcialesDisponibles.push({
                            key: parcialKey,
                            numero: i,
                            data: runner[parcialKey]
                        });
                    }
                }
                
                // Si no hay parciales disponibles, ocultar la sección completa
                if (parcialesDisponibles.length === 0) {
                    parcialesGrid.style.display = 'none';
                } else {
                    parcialesGrid.style.display = 'grid'; // Mostrar la grilla
                    // Generar boxes para cada parcial disponible
                    parcialesDisponibles.forEach(parcial => {
                        const parcialBox = document.createElement('div');
                        parcialBox.className = 'time-box';
                        
                        // Calcular distancia aproximada del parcial
                        const kmDistancia = parcial.numero * 2; // Estimación básica
                        
                        parcialBox.innerHTML = `
                            <div class="time-label">Parcial ${parcial.numero} <small>(aprox. km ${kmDistancia})</small></div>
                            <div class="time-value">${parcial.data.time}</div>
                            <div class="pace-value">${parcial.data.pace}</div>
                            <div class="pace-value">${calculateMinKm(parcial.data.time, parcial.data.pace)}</div>
                        `;
                        parcialesGrid.appendChild(parcialBox);
                    });
                }
                
                // Tiempos finales
                document.getElementById('modal-finish-time').textContent = runner.finishTime?.time || "00:00:00";
                document.getElementById('modal-finish-pace').textContent = runner.finishTime?.pace || "0.00 km/h";
                document.getElementById('modal-gun-time').textContent = runner.gunTime?.time || "00:00:00";
                document.getElementById('modal-gun-pace').textContent = runner.gunTime?.pace || "0.00 km/h";
                
                document.getElementById('modal-finish-minkm').textContent = calculateMinKm(runner.finishTime?.time, runner.finishTime?.pace);
                document.getElementById('modal-gun-minkm').textContent = calculateMinKm(runner.gunTime?.time, runner.gunTime?.pace);
                
                // --- GRÁFICA DE DESEMPEÑO ---
                setTimeout(() => {
                    const ctx = document.getElementById('performanceChart').getContext('2d');
                    if (window.performanceChartInstance) {
                        window.performanceChartInstance.destroy();
                    }
                    // Convertir tiempos a minutos para la gráfica
                    function timeToMinutes(t) {
                        if (!t) return 0;
                        const parts = t.split(':').map(Number);
                        if (parts.length === 3) return parts[0]*60 + parts[1] + parts[2]/60;
                        if (parts.length === 2) return parts[0]*60 + parts[1];
                        return Number(t);
                    }
                    
                    // Crear etiquetas y datos dinámicos para los parciales en la gráfica
                    let labels = [];
                    let runnerData = [];
                    let primerLugarData = [];
                    
                    // Buscar parciales válidos del corredor actual
                    for (let i = 1; i <= 5; i++) {
                        const parcialKey = `mp${i}`;
                        if (runner[parcialKey] && runner[parcialKey].time && runner[parcialKey].time !== "00:00:00") {
                            labels.push(`Parcial ${i}`);
                            runnerData.push(timeToMinutes(runner[parcialKey].time));
                        }
                    }
                    
                    // Agregar tiempos finales siempre
                    labels.push('Tiempo Chip', 'Tiempo oficial');
                    runnerData.push(timeToMinutes(runner.finishTime.time));
                    runnerData.push(timeToMinutes(runner.gunTime.time));

                    // Buscar el primer lugar de la misma distancia y género
                    let primerLugar = resultsData.filter(r => r.distance === runner.distance && r.gender === runner.gender)
                        .sort((a, b) => a.position - b.position)[0];
                    
                    if (primerLugar) {
                        // Buscar parciales válidos del primer lugar
                        for (let i = 1; i <= 5; i++) {
                            const parcialKey = `mp${i}`;
                            if (runner[parcialKey] && runner[parcialKey].time && runner[parcialKey].time !== "00:00:00") {
                                const primerLugarTime = primerLugar[parcialKey] && primerLugar[parcialKey].time !== "00:00:00" 
                                    ? primerLugar[parcialKey].time : "00:00:00";
                                primerLugarData.push(timeToMinutes(primerLugarTime));
                            }
                        }
                        
                        // Agregar tiempos finales del primer lugar
                        primerLugarData.push(timeToMinutes(primerLugar.finishTime.time));
                        primerLugarData.push(timeToMinutes(primerLugar.gunTime.time));
                    } else {
                        // Si no hay primer lugar, llenar con ceros
                        primerLugarData = new Array(runnerData.length).fill(0);
                    }

                    window.performanceChartInstance = new Chart(ctx, {
                        type: 'line',
                        data: {
                            labels: labels,
                            datasets: [
                                {
                                    label: 'Este corredor',
                                    data: runnerData,
                                    borderColor: '#e63946',
                                    backgroundColor: 'rgba(230,57,70,0.1)',
                                    fill: false,
                                    tension: 0.3,
                                    pointRadius: 5,
                                    pointBackgroundColor: '#457b9d',
                                    pointBorderColor: '#1d3557',
                                    pointHoverRadius: 7
                                },
                                {
                                    label: 'Primer lugar',
                                    data: primerLugarData,
                                    borderColor: '#2a9d8f',
                                    backgroundColor: 'rgba(42,157,143,0.1)',
                                    fill: false,
                                    tension: 0.3,
                                    pointRadius: 5,
                                    pointBackgroundColor: '#2a9d8f',
                                    pointBorderColor: '#1d3557',
                                    pointHoverRadius: 7
                                }
                            ]
                        },
                        options: {
                            responsive: true,
                            plugins: {
                                legend: { display: true },
                                title: { display: true, text: 'Gráfica de desempeño (minutos)' }
                            },
                            scales: {
                                y: {
                                    title: { display: true, text: 'Minutos' },
                                    beginAtZero: true
                                }
                            }
                        }
                    });
                }, 100);
                // --- FIN GRÁFICA ---
                
                // Mostrar el modal
                const modal = document.getElementById('runner-details-modal');
                modal.style.display = 'block';
                document.body.style.overflow = 'hidden';
            }
        }
    });
    
    // Cerrar el modal
    document.querySelector('.modal-close').addEventListener('click', function() {
        document.getElementById('runner-details-modal').style.display = 'none';
        document.body.style.overflow = 'auto';
    });
    
    // Cerrar el modal si se hace clic fuera de su contenido
    window.addEventListener('click', function(e) {
        const modal = document.getElementById('runner-details-modal');
        if (e.target === modal) {
            modal.style.display = 'none';
            document.body.style.overflow = 'auto';
        }
    });
    
    // Función para actualizar los botones de paginación
    function updatePaginationButtons() {
        const pagination = document.querySelector('.pagination');
        pagination.innerHTML = '';
        
        // Botón "Anterior"
        const prevButton = document.createElement('button');
        prevButton.innerHTML = '<i class="fas fa-chevron-left"></i>';
        prevButton.disabled = currentPage === 1;
        prevButton.addEventListener('click', () => {
            if (currentPage > 1) {
                currentPage--;
                renderResults(currentFilteredResults);
            }
        });
        pagination.appendChild(prevButton);
        
        // Función para crear botón de página
        function createPageButton(i) {
            const pageButton = document.createElement('button');
            pageButton.textContent = i;
            pageButton.classList.toggle('active', i === currentPage);
            pageButton.addEventListener('click', () => {
                currentPage = i;
                renderResults(currentFilteredResults);
            });
            return pageButton;
        }

        // Función para crear botón de elipsis
        function createEllipsisButton() {
            const ellipsis = document.createElement('span');
            ellipsis.textContent = '...';
            ellipsis.classList.add('pagination-ellipsis');
            return ellipsis;
        }

        // Lógica para mostrar las páginas
        if (totalPages <= 7) {
            // Si hay 7 páginas o menos, mostrar todas
            for (let i = 1; i <= totalPages; i++) {
                pagination.appendChild(createPageButton(i));
            }
        } else {
            // Siempre mostrar primera página
            pagination.appendChild(createPageButton(1));

            // Calcular rango de páginas a mostrar
            let startPage = Math.max(2, currentPage - 2);
            let endPage = Math.min(totalPages - 1, currentPage + 2);

            // Ajustar para mostrar 5 páginas
            if (startPage > 2) pagination.appendChild(createEllipsisButton());

            // Mostrar páginas del rango calculado
            for (let i = startPage; i <= endPage; i++) {
                pagination.appendChild(createPageButton(i));
            }

            // Mostrar elipsis final si es necesario
            if (endPage < totalPages - 1) pagination.appendChild(createEllipsisButton());

            // Siempre mostrar última página
            pagination.appendChild(createPageButton(totalPages));
        }
        
        // Botón "Siguiente"
        const nextButton = document.createElement('button');
        nextButton.innerHTML = '<i class="fas fa-chevron-right"></i>';
        nextButton.disabled = currentPage === totalPages;
        nextButton.addEventListener('click', () => {
            if (currentPage < totalPages) {
                currentPage++;
                renderResults(currentFilteredResults);
            }
        });
        pagination.appendChild(nextButton);
    }
    
    // Para mantener los resultados filtrados actuales
    let currentFilteredResults = [...resultsData];
    
    // Eliminar renderResults(resultsData); para evitar doble renderizado o render vacío
    // El render inicial se hará siempre tras resultsDataLoaded
    // Evento para cambio de distancia
    document.getElementById('filter-distance').addEventListener('change', function() {
        const distance = this.value;
        window.reloadResultsData(distance);
    });
    
    // Evento para mostrar el certificado en una nueva ventana
    document.addEventListener('click', function(e) {
        const certBtn = e.target.closest('.btn-certificado, .card-btn-certificate');
        if (certBtn) {
            e.preventDefault();
            const bibNumber = certBtn.dataset.bib;
            const uniqueId = certBtn.dataset.uniqueid || '';
            
            // Usar solo el ID único para el certificado, como en el botón del modal
            const certUrl = `certificado.html?id=${encodeURIComponent(uniqueId)}`;
            
            window.open(certUrl, '_blank');
        }
    });
    
    // Evento para abrir el modal de fotos desde el botón en el header del modal de detalles
    document.addEventListener('click', function(e) {
        const btnPhotos = e.target.closest('#view-photos, .btn-fotos, .card-btn-photos');
        if (btnPhotos) {
            e.preventDefault();
            let runner;
            
            // Si viene del modal de detalles
            if (btnPhotos.id === 'view-photos') {
                const name = document.getElementById('modal-runner-name').textContent;
                const category = document.getElementById('modal-runner-category').textContent;
                const officialTime = document.getElementById('modal-gun-time').textContent;
                
                // Intentar obtener el UniqueId del modal si está disponible
                const uniqueIdElement = document.getElementById('modal-unique-id');
                const uniqueId = uniqueIdElement ? uniqueIdElement.textContent.trim() : null;
                
                // Buscar por UniqueId primero (más confiable), luego por nombre
                if (uniqueId && uniqueId !== '----') {
                    runner = resultsData.find(r => r.UniqueId === uniqueId);
                } else {
                    // Método alternativo: buscar por nombre directamente en resultsData
                    runner = resultsData.find(r => r.name === name);
                    
                    // Si no se encuentra, usar el método anterior
                    if (!runner) {
                        const row = Array.from(document.querySelectorAll('.runner-name')).find(el => el.textContent.trim() === name)?.closest('tr');
                        runner = row ? resultsData.find(r => r.bib === row.querySelector('.btn-fotos').dataset.bib) : null;
                    }
                }
                
                if (runner) {
                    runner = { ...runner, officialTime };
                }
            } else {
                // Si viene del botón debajo del nombre en la tabla
                const row = btnPhotos.closest('tr');
                const bib = btnPhotos.dataset.bib;
                runner = resultsData.find(r => r.bib === bib);
            }
            
            if (runner) {
                // Validar que el runner tenga la propiedad distance definida
                if (!runner.distance) {
                    console.error('Error: runner.distance no está definido. Runner:', runner);
                    alert('Error: No se pudo determinar la distancia del corredor. Por favor, inténtalo de nuevo.');
                    return;
                }
                
                // Llenar nombre y categoría
                document.getElementById('photos-modal-runner-name').textContent = runner.name;
                document.getElementById('photos-modal-runner-category').textContent = runner.category + ' - ' + runner.distance;
                document.getElementById('photos-modal-runner-number').textContent = 'Corredor numero: ' + runner.bib;
                // console.log(runner);
                
                
                // Asignar fotos con sus tiempos correspondientes usando el startTime de la configuración
                assignPhotosToModal(runner.officialTime || runner.gunTime.time, runner);
                
                document.getElementById('photos-modal').style.display = 'block';
                document.body.style.overflow = 'hidden';
            } else {
                console.error('No se pudo encontrar la información del corredor');
                alert('Error: No se pudo encontrar la información del corredor. Por favor, inténtalo de nuevo.');
            }
        }
    });
    // Cerrar el modal de fotos
    document.querySelector('#photos-modal .modal-close').addEventListener('click', function() {
        document.getElementById('photos-modal').style.display = 'none';
        document.body.style.overflow = 'auto';
    });
    window.addEventListener('click', function(e) {
        const modal = document.getElementById('photos-modal');
        if (e.target === modal) {
            modal.style.display = 'none';
            document.body.style.overflow = 'auto';
        }
    });
    
    // Evento para los botones de expandir/colapsar parciales
    document.addEventListener('click', function(e) {
        const expandBtn = e.target.closest('.expand-details-btn');
        if (expandBtn) {
            const detailsSection = expandBtn.nextElementSibling;
            const isExpanded = detailsSection.classList.toggle('expanded');
            expandBtn.innerHTML = isExpanded ? 
                '<i class="fas fa-chevron-up"></i> Ocultar parciales' : 
                '<i class="fas fa-chevron-down"></i> Ver parciales';
        }
    });
    
    // Evento para cambios de orientación en móvil
    window.addEventListener('orientationchange', function() {
        setTimeout(() => {
            // Re-renderizar los resultados cuando cambia la orientación para ajustar la interfaz
            renderResults(currentFilteredResults);
        }, 200);
    });
});