// Optimizaciones para la vista móvil
document.addEventListener('DOMContentLoaded', function() {
    // Detectar el tipo de dispositivo para aplicar optimizaciones específicas
    const isMobile = window.matchMedia('(max-width: 768px)').matches;
    
    if (isMobile) {
        // Optimizaciones para dispositivos móviles
        applyMobileOptimizations();
    }
    
    // Función para aplicar optimizaciones en dispositivos móviles
    function applyMobileOptimizations() {
        // Optimizar carga de imágenes bajo demanda
        const lazyImages = document.querySelectorAll('.lazy-image');
        if ('IntersectionObserver' in window) {
            const imageObserver = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const img = entry.target;
                        const src = img.getAttribute('data-src');
                        if (src) {
                            img.src = src;
                            img.removeAttribute('data-src');
                            imageObserver.unobserve(img);
                        }
                    }
                });
            });
            
            lazyImages.forEach(img => imageObserver.observe(img));
        } else {
            // Fallback para navegadores que no soportan IntersectionObserver
            lazyImages.forEach(img => {
                const src = img.getAttribute('data-src');
                if (src) img.src = src;
            });
        }
        
        // Agregar botón para expandir/colapsar detalles adicionales
        document.addEventListener('resultsDataLoaded', addExpandDetailsButton);
    }
    
    // Función para agregar botones para expandir detalles en móvil
    function addExpandDetailsButton() {
        const cards = document.querySelectorAll('.results-card');
        cards.forEach(card => {
            // Solo agregar si no existe ya
            if (!card.querySelector('.expand-details-btn')) {
                const expandBtn = document.createElement('button');
                expandBtn.className = 'expand-details-btn';
                expandBtn.innerHTML = '<i class="fas fa-chevron-down"></i> Ver parciales';
                expandBtn.addEventListener('click', function() {
                    const detailsSection = card.querySelector('.details-section');
                    if (detailsSection) {
                        const isExpanded = detailsSection.classList.toggle('expanded');
                        this.innerHTML = isExpanded ? 
                            '<i class="fas fa-chevron-up"></i> Ocultar parciales' : 
                            '<i class="fas fa-chevron-down"></i> Ver parciales';
                    }
                });
                
                // Agregar después de time-info
                const timeInfo = card.querySelector('.time-info');
                if (timeInfo) {
                    // Crear sección de detalles adicionales (parciales)
                    const detailsSection = document.createElement('div');
                    detailsSection.className = 'details-section';
                    
                    // Insertar elementos en el DOM
                    timeInfo.parentNode.insertBefore(expandBtn, timeInfo.nextSibling);
                    timeInfo.parentNode.insertBefore(detailsSection, expandBtn.nextSibling);
                }
            }
        });
    }
    
    // Detectar cambios de orientación
    window.addEventListener('orientationchange', function() {
        setTimeout(() => {
            // Actualizar la interfaz cuando cambia la orientación
            document.dispatchEvent(new CustomEvent('orientationChanged'));
        }, 200); // Pequeño retraso para asegurar que la orientación se ha actualizado
    });
});

// Optimización para tabla de resultados en escritorio
document.addEventListener('DOMContentLoaded', function() {
    const resultsTable = document.querySelector('.results-table');
    if (resultsTable) {
        // Verificar ancho de la pantalla y ajustar la visualización de la tabla
        function adjustTableDisplay() {
            const screenWidth = window.innerWidth;
            
            // En pantallas medianas, añadir indicador de scroll horizontal
            if (screenWidth > 768 && screenWidth < 1200) {
                if (!document.querySelector('.table-scroll-indicator')) {
                    const scrollIndicator = document.createElement('div');
                    scrollIndicator.className = 'table-scroll-indicator';
                    scrollIndicator.innerHTML = '<i class="fas fa-arrows-left-right"></i> Desliza para ver todos los parciales';
                    resultsTable.parentNode.insertBefore(scrollIndicator, resultsTable);
                }
            } else {
                // Eliminar el indicador si la pantalla es suficientemente grande o está en modo móvil
                const scrollIndicator = document.querySelector('.table-scroll-indicator');
                if (scrollIndicator) {
                    scrollIndicator.remove();
                }
            }
        }
        
        // Aplicar al cargar y al cambiar el tamaño de la ventana
        adjustTableDisplay();
        window.addEventListener('resize', adjustTableDisplay);
        document.addEventListener('resultsDataLoaded', adjustTableDisplay);
    }
});
