// Este código establece la integración entre la interfaz de resultados y los certificados
document.addEventListener('DOMContentLoaded', function() {
    // Verificar si estamos en la página de resultados o en la página de certificados
    const isResultsPage = document.getElementById('runner-details-modal');
    const isCertificatePage = document.querySelector('.runner-certificate');
    
    if (isResultsPage) {
        // Estamos en la página de resultados
        configurarInterfazResultados();
    } else if (isCertificatePage) {
        // Estamos en la página de certificados
        cargarDatosCertificado();
    }
});

function configurarInterfazResultados() {
    // Configurar el enlace "Ver reconocimiento"
    const certificateBtn = document.getElementById('view-certificate');
    if (certificateBtn) {
        certificateBtn.addEventListener('click', function(e) {
            e.preventDefault();
            
            // Obtener los datos del corredor desde el modal
            const runnerName = document.getElementById('modal-runner-name').textContent;
            const runnerCategory = document.getElementById('modal-category').textContent;
            const runnerDistance = document.getElementById('modal-distance').textContent;
            const runnerPosition = document.getElementById('modal-position').textContent;
            const runnerChipTime = document.getElementById('modal-finish-time').textContent;
            const runnerOfficialTime = document.getElementById('modal-gun-time').textContent;
            const uniqueId = document.getElementById('modal-unique-id').textContent || '';
            // Obtener el número de dorsal de forma segura
            const runnerBib = document.querySelector('.runner-link')?.dataset.bib || '';
            
            // Crear URL solo con el ID para el certificado (los demás datos se cargarán desde el JSON)
            const certificateUrl = `certificado.html?id=${encodeURIComponent(uniqueId)}`;
            
            // Abrir la página del certificado en una nueva pestaña
            window.open(certificateUrl, '_blank');
        });
    }
}

function cargarDatosCertificado() {
    // Obtener los parámetros de la URL
    const urlParams = new URLSearchParams(window.location.search);
    
    // Verificar si tenemos un ID único
    const uniqueId = urlParams.get('id');
    
    // Si tenemos un ID, el certificado será cargado por certificate-qr.js
    if (uniqueId) {
        // No hacer nada aquí, el otro script se encargará
        return;
    }
    
    // Enfoque anterior (compatibilidad): Obtener datos del corredor desde los parámetros
    const runnerName = urlParams.get('name');
    const runnerCategory = urlParams.get('category');
    const runnerDistance = urlParams.get('distance');
    const runnerPosition = urlParams.get('position');
    const runnerChipTime = urlParams.get('chipTime');
    const runnerOfficialTime = urlParams.get('officialTime');
    const runnerBib = urlParams.get('bib');
    
    // Actualizar el certificado con los datos del corredor
    if (runnerName) document.getElementById('runner-name').textContent = runnerName;
    if (runnerCategory) document.getElementById('runner-category').textContent = runnerCategory;
    if (runnerDistance) document.getElementById('runner-distance').textContent = runnerDistance;
    if (runnerPosition) document.getElementById('runner-position').textContent = runnerPosition + '°';
    if (runnerChipTime) document.getElementById('runner-chip-time').textContent = runnerChipTime;
    if (runnerOfficialTime) document.getElementById('runner-official-time').textContent = runnerOfficialTime;
    
    // Actualizar el ID de verificación
    const verificationText = document.querySelector('.verification-id');
    if (verificationText) {
        if (uniqueId) {
            verificationText.textContent = `ID de verificación: ${runnerBib}`;
            verificationText.setAttribute('data-unique-id', uniqueId);
            
            // Mostrar el ID único en todos los lugares correspondientes
            const uniqueIdDisplay = document.getElementById('unique-id-display');
            if (uniqueIdDisplay) {
                uniqueIdDisplay.textContent = uniqueId;
            }
            
            // Mostrar en el encabezado del certificado
            const headerUniqueId = document.getElementById('header-unique-id');
            if (headerUniqueId) {
                headerUniqueId.textContent = uniqueId;
            }
        } else if (runnerBib) {
            verificationText.textContent = `Número: ${runnerBib}`;
        }
    }
    
    // Configurar el botón de descarga
    const downloadBtn = document.querySelector('.download-button');
    console.log('Botón de descarga encontrado:', downloadBtn);
    if (downloadBtn) {
        downloadBtn.addEventListener('click', function(e) {
            console.log('Botón de descarga clickeado');
            e.preventDefault();
            downloadCertificate();
        });
    } else {
        console.error('No se encontró el botón de descarga');
    }
    
    // Configurar los botones de redes sociales
    const socialButtons = document.querySelectorAll('.social-button');
    socialButtons.forEach(button => {
        button.addEventListener('click', function() {
            const network = this.classList.contains('facebook') ? 'facebook' :
                           this.classList.contains('twitter') ? 'twitter' :
                           this.classList.contains('whatsapp') ? 'whatsapp' :
                           this.classList.contains('instagram') ? 'instagram' : '';
            
            shareOnSocialMedia(network);
        });
    });

    configurarModalEmail();

    if (runnerChipTime && runnerDistance) {
        // Calcular el tiempo promedio por kilómetro
        // runnerChipTime en formato HH:MM:SS, runnerDistance en formato '10k', '5k', etc.
        let distanceNum = parseFloat(runnerDistance);
        if (isNaN(distanceNum)) {
            // Intentar extraer el número si viene como '10K' o '10k'
            const match = runnerDistance.match(/(\d+(?:\.\d+)?)/);
            if (match) distanceNum = parseFloat(match[1]);
        }
        if (distanceNum > 0) {
            const [h, m, s] = runnerChipTime.split(':').map(Number);
            const totalSeconds = h * 3600 + m * 60 + s;
            const paceSeconds = totalSeconds / distanceNum;
            const paceMin = Math.floor(paceSeconds / 60);
            const paceSec = Math.round(paceSeconds % 60).toString().padStart(2, '0');
            document.getElementById('runner-pace').textContent = `${paceMin}:${paceSec} min/km`;
        }
    }
}

function configurarModalEmail() {
    const emailButton = document.querySelector('.email-button');
    const emailModal = document.getElementById('email-modal');
    const modalClose = emailModal.querySelector('.modal-close');
    const sendEmailButton = emailModal.querySelector('.send-email-button');
    const emailInput = emailModal.querySelector('#email');

    // Abrir modal
    emailButton.addEventListener('click', function(e) {
        e.preventDefault();
        emailModal.classList.add('show');
    });

    // Cerrar modal
    modalClose.addEventListener('click', function() {
        emailModal.classList.remove('show');
    });

    // Cerrar modal al hacer clic fuera
    window.addEventListener('click', function(e) {
        if (e.target === emailModal) {
            emailModal.classList.remove('show');
        }
    });

    // Enviar correo
    sendEmailButton.addEventListener('click', async function() {
        const email = emailInput.value.trim();
        if (!email) {
            alert('Por favor, ingresa un correo electrónico válido');
            return;
        }

        const subject = 'Tu certificado de participación';
        // Clonar el certificado y eliminar los botones antes de enviar
        const certificateClone = document.querySelector('.runner-certificate').cloneNode(true);
        const downloadBtn = certificateClone.querySelector('.download-button');
        if (downloadBtn) downloadBtn.remove();
        const emailBtn = certificateClone.querySelector('.email-button');
        if (emailBtn) emailBtn.remove();
        const htmlContent = certificateClone.outerHTML;

        sendEmailButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Enviando...';

        try {
            const response = await fetch('send-certificate.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ email, subject, htmlContent })
            });
            const result = await response.json();
            if (result.success) {
                alert('¡Certificado enviado con éxito!');
                emailModal.classList.remove('show');
                emailInput.value = '';
            } else {
                throw new Error(result.error);
            }
        } catch (error) {
            alert('Error al enviar el certificado. Por favor, intenta nuevamente.');
        } finally {
            sendEmailButton.innerHTML = '<i class="fas fa-paper-plane"></i> Enviar';
        }
    });
}