// Función para cargar los datos del archivo JSON
async function loadResultsData(filename = 'data_10k.json') {
    try {
        // Añadir parámetro de timestamp para evitar el caché del navegador
        const cacheBuster = `?t=${Date.now()}`;
        const response = await fetch(`data/${filename}${cacheBuster}`);
        
        // Verificar que la respuesta fue exitosa
        if (!response.ok) {
            console.error(`Error al cargar ${filename}: ${response.status} ${response.statusText}`);
            return [];
        }
        
        const data = await response.json();
        
        // Validar que data es un array
        if (!Array.isArray(data)) {
            console.error(`Error: ${filename} no contiene un array válido`);
            return [];
        }
        
        // Registrar información completa del primer corredor para depuración
        if (data.length > 0) {
            console.log('Datos JSON cargados para', filename);
            console.log('Primer corredor:', data[0]);
            console.log('¿Tiene UniqueId?', data[0].hasOwnProperty('UniqueId'));
            if (data[0].UniqueId) {
                console.log('Valor de UniqueId:', data[0].UniqueId);
            } else {
                console.error('⚠️ ADVERTENCIA: No se encontró UniqueId en los datos JSON');
            }
        } else {
            console.warn('No se encontraron datos en el archivo JSON');
            return [];
        }
        
        // Identificar las columnas de parciales (todas las columnas entre Bib# y GunTime)
        const parcialesKeys = [];
        if (data.length > 0) {
            const sampleRunner = data[0];
            const keys = Object.keys(sampleRunner);
            let isBibFound = false;
            let bibKeyName = '';
            
            // Primero identificar la clave del dorsal, que puede ser "Bib" o "Bib#"
            for (const key of keys) {
                if (key === "Bib" || key === "Bib#") {
                    bibKeyName = key;
                    break;
                }
            }
            
            // Si no se encontró ningún campo de dorsal, usar el primer campo como referencia
            if (!bibKeyName && keys.length > 0) {
                bibKeyName = keys[0];
                console.warn("No se encontró un campo de dorsal. Usando el primer campo como referencia:", bibKeyName);
            }
            
            // Ahora buscar parciales
            for (const key of keys) {
                if (key === bibKeyName) {
                    isBibFound = true;
                    continue;
                }
                
                // Excluir campos que no son parciales
                const excludedFields = ["GunTime", "FinishTime", "UniqueId", "Category", "Race", "Gender", "Athlete", "Pos"];
                if (isBibFound && !excludedFields.includes(key)) {
                    parcialesKeys.push(key);
                }
                
                if (key === "GunTime") {
                    break;
                }
            }
            
            console.log("Parciales detectados:", parcialesKeys);
        }
        
        // Transformar los datos al formato esperado por la aplicación, con parciales dinámicos
        const mappedData = data.map(runner => {
            // Configuración para la distancia basada en el nombre del archivo
            // IMPORTANTE: Ordenar de mayor a menor especificidad para evitar coincidencias parciales
            let distanceValue = '10k';
            let totalDistance = 10;
            
            if (filename.includes('50k')) {
                distanceValue = '50k';
                totalDistance = 50;
            } else if (filename.includes('42k')) {
                distanceValue = '42k';
                totalDistance = 42;
            } else if (filename.includes('25k')) {
                distanceValue = '25k';
                totalDistance = 25;
            } else if (filename.includes('21k')) {
                distanceValue = '21k';
                totalDistance = 21;
            } else if (filename.includes('15k')) {
                distanceValue = '15k';
                totalDistance = 15;
            } else if (filename.includes('10k')) {
                distanceValue = '10k';
                totalDistance = 10;
            } else if (filename.includes('5k')) {
                distanceValue = '5k';
                totalDistance = 5;
            } else if (filename.includes('3k')) {
                distanceValue = '3k';
                totalDistance = 3;
            }
            
            // Normalizar el acceso a campos comunes
            const bibValue = runner.Bib || runner["Bib#"] || "";
            const nameValue = runner.Athlete || runner.Name || "";
            const categoryValue = runner.Category || runner.Class || "";
            
            // Crear un objeto base
            const runnerObj = {
                position: runner.Pos,
                name: nameValue,
                bib: bibValue.toString(),
                category: categoryValue,
                gender: runner.Gender,
                distance: distanceValue,
                finishTime: {time: runner.FinishTime, pace: calculatePace(runner.FinishTime, totalDistance)},
                gunTime: {time: runner.GunTime, pace: calculatePace(runner.GunTime, totalDistance)},
                UniqueId: runner.UniqueId
            };
            
            // Definir parciales estáticos predeterminados en caso de no existir parciales dinámicos
            const defaultParciales = {
                mp1: {time: "00:00:00", pace: "0 km/h"},
                mp2: {time: "00:00:00", pace: "0 km/h"},
                mp3: {time: "00:00:00", pace: "0 km/h"},
                mp4: {time: "00:00:00", pace: "0 km/h"},
                mp5: {time: "00:00:00", pace: "0 km/h"}
            };
            
            // Si hay parciales definidos, procesarlos dinámicamente
            if (parcialesKeys.length > 0) {
                // Distancias para los parciales (asumidas como proporcionales)
                const parcialDistances = {};
                parcialesKeys.forEach((key, index) => {
                    // Extraer valor numérico de la clave si es posible
                    let distance = 0;
                    const keyMatch = key.match(/(\d+(?:\.\d+)?)/);
                    if (keyMatch) {
                        distance = parseFloat(keyMatch[1]);
                    } else {
                        // Si no se puede extraer, usar proporciones
                        distance = Math.round((index + 1) * (totalDistance / (parcialesKeys.length + 1)));
                    }
                    parcialDistances[key] = distance;
                });
                
                // Mapear parciales al formato esperado (hasta 5 máximo)
                const parcialKeys = Object.keys(parcialDistances).slice(0, 5);
                parcialKeys.forEach((key, index) => {
                    const mpKey = `mp${index + 1}`;
                    const time = runner[key] || "00:00:00";
                    const distance = parcialDistances[key];
                    runnerObj[mpKey] = {
                        time: time, 
                        pace: calculatePace(time, distance)
                    };
                });
                
                // Llenar los parciales restantes si hay menos de 5
                for (let i = parcialKeys.length + 1; i <= 5; i++) {
                    runnerObj[`mp${i}`] = defaultParciales[`mp${i}`];
                }
            } else {
                // Si no se detectaron parciales, usar parciales predeterminados del formato antiguo
                runnerObj.mp1 = {time: runner.MP1 || "00:00:00", pace: calculatePace(runner.MP1 || "00:00:00", 2)};
                runnerObj.mp2 = {time: runner.MP2 || "00:00:00", pace: calculatePace(runner.MP2 || "00:00:00", 4)};
                runnerObj.mp3 = {time: runner.MP3 || "00:00:00", pace: calculatePace(runner.MP3 || "00:00:00", 6)};
                runnerObj.mp4 = {time: runner.MP4 || "00:00:00", pace: calculatePace(runner.MP4 || "00:00:00", 8)};
                runnerObj.mp5 = {time: runner.MP5 || "00:00:00", pace: calculatePace(runner.MP5 || "00:00:00", 9)};
            }
            
            return runnerObj;
        });
        
        // Verificar que los datos mapeados incluyen UniqueId
        if (mappedData.length > 0) {
            console.log('Datos mapeados con UniqueId:', mappedData[0].UniqueId);
        }
        
        // Exponer la información de los parciales detectados para que otros scripts puedan usarla
        window.parcialesInfo = {
            keys: parcialesKeys,
            labels: parcialesKeys.map(key => {
                // Extraer el valor numérico para mostrarlo como etiqueta
                const match = key.match(/(\d+(?:\.\d+)?)/);
                if (match) {
                    return `${match[1]} km`;
                } 
                return key;
            }),
            count: parcialesKeys.length
        };
        
        return mappedData;
    } catch (error) {
        console.error('Error al cargar los datos:', error);
        return [];
    }
}

// Función para calcular el ritmo (pace) en km/h
function calculatePace(time, distance) {
    if (!time) return "0 km/h";
    
    const [hours, minutes, seconds] = time.split(':').map(Number);
    const totalSeconds = hours * 3600 + minutes * 60 + seconds;
    const pace = (distance / totalSeconds) * 3600; // km/h
    
    return pace.toFixed(2) + " km/h";
}

// Variables globales para almacenar los datos
let resultsData = [];
let currentFilteredResults = [];

// Función para recargar los datos según la distancia
window.reloadResultsData = async function(distance) {
    console.log('Recargando datos para distancia:', distance);
    let files = [];
    if (!distance) {
        files = ['data_5k.json', 'data_10k.json', 'data_21k.json'];
    } else if (distance === '5k') {
        files = ['data_5k.json'];
    } else if (distance === '10k') {
        files = ['data_10k.json'];
    } else if (distance === '21k') {
        files = ['data_21k.json'];
    } else {
        files = ['data_10k.json'];
    }
    
    console.log('Archivos a cargar:', files);
    
    let allData = [];
    for (const file of files) {
        console.log('Cargando archivo:', file);
        const data = await loadResultsData(file);
        console.log('Datos cargados desde', file, ':', data.length, 'registros');
        allData = allData.concat(data);
    }
    
    console.log('Total de datos cargados:', allData.length, 'registros');
    resultsData = allData;
    currentFilteredResults = [...allData];
    document.dispatchEvent(new CustomEvent('resultsDataLoaded'));
}

// Cargar los datos cuando se inicie la aplicación (por defecto 10k)
document.addEventListener('DOMContentLoaded', function() {
    loadResultsData('data_10k.json').then(data => {
        resultsData = data;
        currentFilteredResults = [...data];
        // Disparar un evento personalizado cuando los datos estén listos
        document.dispatchEvent(new CustomEvent('resultsDataLoaded'));
    });
});