/**
 * Script para el panel de administración
 * Gestiona la carga y validación de archivos JSON
 */

document.addEventListener('DOMContentLoaded', function() {
    const fileInput = document.getElementById('jsonFile');
    const distanceSelect = document.getElementById('distance');
    const uploadBtn = document.getElementById('uploadBtn');
    const statusMessage = document.getElementById('statusMessage');
    const filePreview = document.getElementById('filePreview');
    const loader = document.getElementById('loader');
    
    // Mostrar vista previa del archivo JSON
    fileInput.addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (!file) {
            filePreview.style.display = 'none';
            return;
        }
        
        const reader = new FileReader();
        reader.onload = function(e) {
            try {
                // Intenta parsear el JSON para validar
                const json = JSON.parse(e.target.result);
                // Muestra una versión formateada del JSON (primeros 200 caracteres)
                const jsonString = JSON.stringify(json, null, 2);
                filePreview.textContent = jsonString.length > 200 ? 
                    jsonString.substring(0, 200) + '...' : jsonString;
                filePreview.style.display = 'block';
            } catch (error) {
                showStatus('El archivo seleccionado no es un JSON válido.', 'error');
                filePreview.style.display = 'none';
            }
        };
        reader.readAsText(file);
    });
    
    // Subir archivo
    uploadBtn.addEventListener('click', function() {
        const file = fileInput.files[0];
        const distance = distanceSelect.value;
        
        // Validaciones
        if (!distance) {
            showStatus('Por favor selecciona una distancia.', 'error');
            return;
        }
        
        if (!file) {
            showStatus('Por favor selecciona un archivo JSON.', 'error');
            return;
        }
        
        // Crear FormData para enviar el archivo
        const formData = new FormData();
        formData.append('jsonFile', file);
        formData.append('distance', distance);
        
        // Mostrar loader
        loader.style.display = 'block';
        statusMessage.style.display = 'none';
        
        // Enviar archivo al servidor
        fetch('php/upload-json.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            loader.style.display = 'none';
            if (data.success) {
                showStatus(data.message, 'success');
                // Limpiar campos
                fileInput.value = '';
                distanceSelect.value = '';
                filePreview.style.display = 'none';
            } else {
                showStatus(data.message, 'error');
            }
        })
        .catch(error => {
            loader.style.display = 'none';
            showStatus('Error al subir el archivo: ' + error.message, 'error');
        });
    });
    
    function showStatus(message, type) {
        statusMessage.textContent = message;
        statusMessage.className = 'status ' + type;
        statusMessage.style.display = 'block';
    }
});
