// Importar la configuración
import CONFIG from '../../config.js';
import { assignPhotosToModal } from './photo-timing.js';

// Scripts para la búsqueda y filtrado de resultados
document.addEventListener('DOMContentLoaded', function() {
    const ITEMS_PER_PAGE = 10;
    let currentPage = 1;
    let totalPages = 0;
    
    // Esperar a que los datos se carguen
    document.addEventListener('resultsDataLoaded', function() {
        updateCategorySelect();
        totalPages = Math.ceil(currentFilteredResults.length / ITEMS_PER_PAGE);
        renderResults(currentFilteredResults);
    });
    
    // Función para actualizar el select de categorías con los datos cargados
    function updateCategorySelect() {
        const categorySelect = document.getElementById('filter-category');
        const currentValue = categorySelect.value; // Preservar la selección actual
        
        // Obtener categorías únicas de los datos
        const categories = [...new Set(resultsData.map(runner => runner.category))].filter(Boolean).sort();
        
        // Limpiar opciones existentes excepto "Todas las categorías"
        categorySelect.innerHTML = '<option value="">Todas las categorías</option>';
        
        // Agregar las categorías encontradas en los datos
        categories.forEach(category => {
            const option = document.createElement('option');
            option.value = category;
            option.textContent = category;
            categorySelect.appendChild(option);
        });
        
        // Restaurar la selección anterior si aún existe
        if (currentValue && categories.includes(currentValue)) {
            categorySelect.value = currentValue;
        }
        
        console.log('Categorías encontradas:', categories);
    }
    
    // Función para renderizar los resultados en la tabla con paginación
    function renderResults(results) {
        const tbody = document.querySelector('.results-table tbody');
        tbody.innerHTML = '';
        
        if (!results || results.length === 0) {
            const row = document.createElement('tr');
            row.innerHTML = '<td colspan="6" class="no-results">No se encontraron resultados que coincidan con tu búsqueda</td>';
            tbody.appendChild(row);
            document.getElementById('results-count').textContent = '0';
            document.getElementById('total-results').textContent = '0';
            // Ocultar paginación
            document.querySelector('.pagination').style.display = 'none';
            return;
        }
        
        // Mostrar paginación
        document.querySelector('.pagination').style.display = 'flex';
        
        // Calcular índices de inicio y fin para la página actual
        const startIndex = (currentPage - 1) * ITEMS_PER_PAGE;
        const endIndex = Math.min(startIndex + ITEMS_PER_PAGE, results.length);
        
        // Mostrar solo los resultados de la página actual
        for (let i = startIndex; i < endIndex; i++) {
            const runner = results[i];
            const row = document.createElement('tr');
            const isTop3 = runner.position <= 3 ? 'top-3' : '';
            
            // Agregar clase a la fila si es top 3
            if (runner.position <= 3) {
                row.classList.add('top-3-row');
            }
            
            row.innerHTML = `
                <td class="position ${isTop3}">${runner.position ? runner.position + '°' : 'No definido'}</td>
                <td>
                    <div class="runner-details">
                        <span class="bib-number">${runner.bib || 'No definido'}</span>
                        <div>
                            <div class="runner-name"><a href="#" class="runner-link" data-bib="${runner.bib}" data-uniqueid="${runner.UniqueId || ''}">${runner.name || 'No definido'}</a></div>
                            <div class="runner-category">${runner.category || 'No definido'} - ${runner.distance || 'No definido'}</div>
                            <div class="actions-cell-horizontal" style="margin: 4px 0 0 0;">
                                <button class="action-btn-rect btn-certificado" title="Ver certificado" data-bib="${runner.bib}" data-uniqueid="${runner.UniqueId || ''}"><i class="fas fa-certificate"></i></button>
                                <button class="action-btn-rect btn-metricas" title="Ver métricas" data-bib="${runner.bib}" data-uniqueid="${runner.UniqueId || ''}"><i class="fas fa-chart-bar"></i></button>
                                <button class="action-btn-rect btn-fotos" title="Ver fotos" data-bib="${runner.bib}" data-uniqueid="${runner.UniqueId || ''}"><i class="fas fa-camera"></i></button>
                            </div>
                        </div>
                    </div>
                </td>
                <td>
                    <div class="time">${runner.mp1.time || 'No definido'}</div>
                    <div class="pace">${runner.mp1.pace || 'No definido'}</div>
                </td>
                <td>
                    <div class="time">${runner.mp2.time || 'No definido'}</div>
                    <div class="pace">${runner.mp2.pace || 'No definido'}</div>
                </td>
                <td>
                    <div class="time">${runner.finishTime.time || 'No definido'}</div>
                    <div class="pace">${runner.finishTime.pace || 'No definido'}</div>
                </td>
                <td>
                    <div class="time">${runner.gunTime.time || 'No definido'}</div>
                    <div class="pace">${runner.gunTime.pace || 'No definido'}</div>
                </td>
            `;
            
            tbody.appendChild(row);
        }
        
        // Actualizar contador de resultados
        document.getElementById('results-count').textContent = Math.min(ITEMS_PER_PAGE, results.length - startIndex);
        document.getElementById('total-results').textContent = results.length;
        
        // Calcular total de páginas
        totalPages = Math.ceil(results.length / ITEMS_PER_PAGE);
        
        // Asegurar que currentPage no supere el total de páginas
        if (currentPage > totalPages) {
            currentPage = 1;
        }
        
        // Actualizar los botones de paginación
        updatePaginationButtons();
    }
    
    // Función para filtrar resultados
    function filterResults() {
        const nameSearch = document.getElementById('search-name').value.toLowerCase();
        const bibSearch = document.getElementById('search-bib').value;
        const categoryFilter = document.getElementById('filter-category').value;
        const distanceFilter = document.getElementById('filter-distance').value;
        const genderFilter = document.getElementById('filter-gender').value;
        
        currentFilteredResults = resultsData.filter(runner => {
            // Filtro por nombre
            if (nameSearch && !runner.name.toLowerCase().includes(nameSearch)) {
                return false;
            }
            
            // Filtro por dorsal
            if (bibSearch && runner.bib !== bibSearch) {
                return false;
            }
            
            // Filtro por categoría
            if (categoryFilter && runner.category !== categoryFilter) {
                return false;
            }
            
            // Filtro por distancia
            if (distanceFilter && runner.distance !== distanceFilter) {
                return false;
            }
            
            // Filtro por género
            if (genderFilter && runner.gender !== genderFilter) {
                return false;
            }
            
            return true;
        });
        
        // Resetear a la primera página cuando se aplica un filtro
        currentPage = 1;
        renderResults(currentFilteredResults);
    }
    
    // Evento para botón de búsqueda
    document.getElementById('search-btn').addEventListener('click', filterResults);
    
    // Evento para tecla Enter en los campos de búsqueda
    document.getElementById('search-name').addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            filterResults();
        }
    });
    
    document.getElementById('search-bib').addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            filterResults();
        }
    });
    
    // Botón de reset
    document.getElementById('reset-btn').addEventListener('click', function() {
        document.getElementById('search-name').value = '';
        document.getElementById('search-bib').value = '';
        document.getElementById('filter-category').value = '';
        document.getElementById('filter-distance').value = '';
        document.getElementById('filter-gender').value = '';
        currentFilteredResults = [...resultsData];
        currentPage = 1;
        renderResults(currentFilteredResults);
    });
    
    // Evento para ordenar resultados
    document.getElementById('sort-select').addEventListener('change', function() {
        const sortBy = this.value;
        let sortedResults = [...currentFilteredResults];
        
        switch(sortBy) {
            case 'position':
                sortedResults.sort((a, b) => a.position - b.position);
                break;
            case 'name':
                sortedResults.sort((a, b) => a.name.localeCompare(b.name));
                break;
            case 'time':
                sortedResults.sort((a, b) => a.finishTime.time.localeCompare(b.finishTime.time));
                break;
            case 'category':
                sortedResults.sort((a, b) => a.category.localeCompare(b.category));
                break;
        }
        
        currentFilteredResults = sortedResults;
        renderResults(currentFilteredResults);
    });
    
    // Manejo del modal de detalles de corredor
    document.addEventListener('click', function(e) {
        const runnerLink = e.target.closest('.runner-link, .btn-metricas');
        if (runnerLink) {
            e.preventDefault();
            const bibNumber = runnerLink.dataset.bib;
            const runner = resultsData.find(r => r.bib === bibNumber);
            if (runner) {
                // Rellenar los datos del corredor en el modal
                document.getElementById('modal-runner-name').textContent = runner.name;
                document.getElementById('modal-runner-category').textContent = `${runner.distance} - ${runner.gender === 'M' ? 'Varonil' : 'Femenil'} - ${runner.category}`;
                document.getElementById('modal-distance').textContent = runner.distance;
                document.getElementById('modal-gender').textContent = runner.gender;
                document.getElementById('modal-category').textContent = runner.category;
                document.getElementById('modal-position').textContent = runner.position;
                
                // Mostrar ID único de verificación
                const uniqueIdElement = document.getElementById('modal-unique-id');
                if (uniqueIdElement) {
                    uniqueIdElement.textContent = runner.UniqueId || '----';
                }
                
                // Calcular el total real de corredores para esta distancia
                const runnersForDistance = resultsData.filter(r => r.distance === runner.distance);
                
                const totalRunners = runnersForDistance.length;
                
                document.getElementById('modal-total-runners').textContent = `${totalRunners} corredores`;
                document.getElementById('modal-runners-before').textContent = runner.position - 1;
                document.getElementById('modal-runners-after').textContent = totalRunners - runner.position;
                
                // Tiempos - ocultar parciales que no existen
                const mp1Box = document.querySelector('.times-grid .time-box:nth-child(1)');
                const mp2Box = document.querySelector('.times-grid .time-box:nth-child(2)');
                
                if (runner.mp1 && runner.mp1.time !== "00:00:00") {
                    document.getElementById('modal-mp1-time').textContent = runner.mp1.time;
                    document.getElementById('modal-mp1-pace').textContent = runner.mp1.pace;
                    document.getElementById('modal-mp1-minkm').textContent = "00:03:" + (Math.floor(Math.random() * 30) + 10).toString().padStart(2, '0') + " min/km";
                    if (mp1Box) mp1Box.style.display = 'block';
                } else {
                    if (mp1Box) mp1Box.style.display = 'none';
                }
                
                if (runner.mp2 && runner.mp2.time !== "00:00:00") {
                    document.getElementById('modal-mp2-time').textContent = runner.mp2.time;
                    document.getElementById('modal-mp2-pace').textContent = runner.mp2.pace;
                    document.getElementById('modal-mp2-minkm').textContent = "00:03:" + (Math.floor(Math.random() * 30) + 10).toString().padStart(2, '0') + " min/km";
                    if (mp2Box) mp2Box.style.display = 'block';
                } else {
                    if (mp2Box) mp2Box.style.display = 'none';
                }
                
                document.getElementById('modal-finish-time').textContent = runner.finishTime.time;
                document.getElementById('modal-finish-pace').textContent = runner.finishTime.pace;
                document.getElementById('modal-gun-time').textContent = runner.gunTime.time;
                document.getElementById('modal-gun-pace').textContent = runner.gunTime.pace;
                
                document.getElementById('modal-finish-minkm').textContent = "00:03:" + (Math.floor(Math.random() * 30) + 10).toString().padStart(2, '0') + " min/km";
                document.getElementById('modal-gun-minkm').textContent = "00:03:" + (Math.floor(Math.random() * 30) + 10).toString().padStart(2, '0') + " min/km";
                
                // --- GRÁFICA DE DESEMPEÑO ---
                setTimeout(() => {
                    const ctx = document.getElementById('performanceChart').getContext('2d');
                    if (window.performanceChartInstance) {
                        window.performanceChartInstance.destroy();
                    }
                    // Convertir tiempos a minutos para la gráfica
                    function timeToMinutes(t) {
                        if (!t) return 0;
                        const parts = t.split(':').map(Number);
                        if (parts.length === 3) return parts[0]*60 + parts[1] + parts[2]/60;
                        if (parts.length === 2) return parts[0]*60 + parts[1];
                        return Number(t);
                    }
                    const labels = ['Parcial 1', 'Parcial 2', 'Tiempo Chip', 'Tiempo oficial'];
                    const data = [runner.mp1.time, runner.mp2.time, runner.finishTime.time, runner.gunTime.time].map(timeToMinutes);

                    // Buscar el primer lugar de la misma distancia y género
                    let primerLugar = resultsData.filter(r => r.distance === runner.distance && r.gender === runner.gender)
                        .sort((a, b) => a.position - b.position)[0];
                    let dataPrimerLugar = [0,0,0,0];
                    if (primerLugar) {
                        dataPrimerLugar = [primerLugar.mp1.time, primerLugar.mp2.time, primerLugar.finishTime.time, primerLugar.gunTime.time].map(timeToMinutes);
                    }

                    window.performanceChartInstance = new Chart(ctx, {
                        type: 'line',
                        data: {
                            labels: labels,
                            datasets: [
                                {
                                    label: 'Este corredor',
                                    data: data,
                                    borderColor: '#e63946',
                                    backgroundColor: 'rgba(230,57,70,0.1)',
                                    fill: false,
                                    tension: 0.3,
                                    pointRadius: 5,
                                    pointBackgroundColor: '#457b9d',
                                    pointBorderColor: '#1d3557',
                                    pointHoverRadius: 7
                                },
                                {
                                    label: 'Primer lugar',
                                    data: dataPrimerLugar,
                                    borderColor: '#2a9d8f',
                                    backgroundColor: 'rgba(42,157,143,0.1)',
                                    fill: false,
                                    tension: 0.3,
                                    pointRadius: 5,
                                    pointBackgroundColor: '#2a9d8f',
                                    pointBorderColor: '#1d3557',
                                    pointHoverRadius: 7
                                }
                            ]
                        },
                        options: {
                            responsive: true,
                            plugins: {
                                legend: { display: true },
                                title: { display: true, text: 'Gráfica de desempeño (minutos)' }
                            },
                            scales: {
                                y: {
                                    title: { display: true, text: 'Minutos' },
                                    beginAtZero: true
                                }
                            }
                        }
                    });
                }, 100);
                // --- FIN GRÁFICA ---
                
                // Mostrar el modal
                const modal = document.getElementById('runner-details-modal');
                modal.style.display = 'block';
                document.body.style.overflow = 'hidden';
            }
        }
    });
    
    // Cerrar el modal
    document.querySelector('.modal-close').addEventListener('click', function() {
        document.getElementById('runner-details-modal').style.display = 'none';
        document.body.style.overflow = 'auto';
    });
    
    // Cerrar el modal si se hace clic fuera de su contenido
    window.addEventListener('click', function(e) {
        const modal = document.getElementById('runner-details-modal');
        if (e.target === modal) {
            modal.style.display = 'none';
            document.body.style.overflow = 'auto';
        }
    });
    
    // Función para actualizar los botones de paginación
    function updatePaginationButtons() {
        const pagination = document.querySelector('.pagination');
        pagination.innerHTML = '';
        
        // Botón "Anterior"
        const prevButton = document.createElement('button');
        prevButton.innerHTML = '<i class="fas fa-chevron-left"></i>';
        prevButton.disabled = currentPage === 1;
        prevButton.addEventListener('click', () => {
            if (currentPage > 1) {
                currentPage--;
                renderResults(currentFilteredResults);
            }
        });
        pagination.appendChild(prevButton);
        
        // Números de página
        for (let i = 1; i <= totalPages; i++) {
            const pageButton = document.createElement('button');
            pageButton.textContent = i;
            pageButton.classList.toggle('active', i === currentPage);
            pageButton.addEventListener('click', () => {
                currentPage = i;
                renderResults(currentFilteredResults);
            });
            pagination.appendChild(pageButton);
        }
        
        // Botón "Siguiente"
        const nextButton = document.createElement('button');
        nextButton.innerHTML = '<i class="fas fa-chevron-right"></i>';
        nextButton.disabled = currentPage === totalPages;
        nextButton.addEventListener('click', () => {
            if (currentPage < totalPages) {
                currentPage++;
                renderResults(currentFilteredResults);
            }
        });
        pagination.appendChild(nextButton);
    }
    
    // Para mantener los resultados filtrados actuales
    let currentFilteredResults = [...resultsData];
    
    // Renderizar los resultados iniciales
    renderResults(resultsData);
    
    // Evento para cambio de distancia
    document.getElementById('filter-distance').addEventListener('change', function() {
        const distance = this.value;
        window.reloadResultsData(distance);
        // Aplicar actualización de categorías después de cargar los nuevos datos
        document.addEventListener('resultsDataLoaded', function onDataLoaded() {
            updateCategorySelect(); // Actualizar categorías con los nuevos datos
            // Remover el listener para evitar duplicados
            document.removeEventListener('resultsDataLoaded', onDataLoaded);
        });
    });
    
    // Evento para mostrar el certificado en una nueva ventana
    document.addEventListener('click', function(e) {
        const certBtn = e.target.closest('.btn-certificado');
        if (certBtn) {
            e.preventDefault();
            const bibNumber = certBtn.dataset.bib;
            const uniqueId = certBtn.dataset.uniqueid || '';
            
            // Usar solo el ID único para el certificado, como en el botón del modal
            const certUrl = `certificado.html?id=${encodeURIComponent(uniqueId)}`;
            
            window.open(certUrl, '_blank');
        }
    });
    
    // Evento para abrir el modal de fotos desde el botón en el header del modal de detalles
    document.addEventListener('click', function(e) {
        const btnPhotos = e.target.closest('#view-photos, .btn-fotos');
        if (btnPhotos) {
            e.preventDefault();
            let runner;
            
            // Si viene del modal de detalles
            if (btnPhotos.id === 'view-photos') {
                const name = document.getElementById('modal-runner-name').textContent;
                const category = document.getElementById('modal-runner-category').textContent;
                const officialTime = document.getElementById('modal-gun-time').textContent;
                const row = Array.from(document.querySelectorAll('.runner-name')).find(el => el.textContent.trim() === name)?.closest('tr');
                runner = row ? resultsData.find(r => r.bib === row.querySelector('.btn-fotos').dataset.bib) : null;
                runner = { ...runner, officialTime };
            } else {
                // Si viene del botón debajo del nombre en la tabla
                const row = btnPhotos.closest('tr');
                const bib = btnPhotos.dataset.bib;
                runner = resultsData.find(r => r.bib === bib);
            }
            
            if (runner) {
                // Llenar nombre y categoría
                document.getElementById('photos-modal-runner-name').textContent = runner.name;
                document.getElementById('photos-modal-runner-category').textContent = runner.category + ' - ' + runner.distance;
                document.getElementById('photos-modal-runner-number').textContent = 'Corredor numero: ' + runner.bib;
                console.log(runner);
                
                
                // Asignar fotos con sus tiempos correspondientes usando el startTime de la configuración
                assignPhotosToModal(runner.officialTime || runner.gunTime.time, runner);
                
                document.getElementById('photos-modal').style.display = 'block';
                document.body.style.overflow = 'hidden';
            }
        }
    });
    // Cerrar el modal de fotos
    document.querySelector('#photos-modal .modal-close').addEventListener('click', function() {
        document.getElementById('photos-modal').style.display = 'none';
        document.body.style.overflow = 'auto';
    });
    window.addEventListener('click', function(e) {
        const modal = document.getElementById('photos-modal');
        if (e.target === modal) {
            modal.style.display = 'none';
            document.body.style.overflow = 'auto';
        }
    });
});