// Variables globales para la galería
let currentPhotoIndex = 0;
let galleryPhotos = [];
let touchStartX = 0;
let touchEndX = 0;

// Función para inicializar la galería
function initGallery(photos) {
    galleryPhotos = photos;
    currentPhotoIndex = 0;
    updateGalleryView();
    setupGalleryThumbnails();
    setupTouchEvents();
}

// Función para actualizar la vista de la galería
function updateGalleryView() {
    const currentPhoto = galleryPhotos[currentPhotoIndex];
    const galleryImg = document.getElementById('gallery-current-photo');
    const galleryTime = document.getElementById('gallery-current-time');
    const galleryCounter = document.getElementById('gallery-counter');
    
    galleryImg.src = currentPhoto.src;
    galleryTime.textContent = currentPhoto.time;
    galleryCounter.textContent = `${currentPhotoIndex + 1}/${galleryPhotos.length}`;
    
    // Actualizar miniaturas activas
    document.querySelectorAll('.gallery-thumbnail').forEach((thumb, index) => {
        thumb.classList.toggle('active', index === currentPhotoIndex);
    });

    // Animar la transición de la imagen
    galleryImg.style.opacity = '0';
    setTimeout(() => {
        galleryImg.style.opacity = '1';
    }, 50);
}

// Función para configurar las miniaturas
function setupGalleryThumbnails() {
    const thumbnailsContainer = document.querySelector('.gallery-thumbnails');
    thumbnailsContainer.innerHTML = '';
    
    galleryPhotos.forEach((photo, index) => {
        const thumb = document.createElement('img');
        thumb.src = photo.src;
        thumb.classList.add('gallery-thumbnail');
        if (index === currentPhotoIndex) {
            thumb.classList.add('active');
        }
        thumb.addEventListener('click', () => {
            currentPhotoIndex = index;
            updateGalleryView();
        });
        thumbnailsContainer.appendChild(thumb);
    });
}

// Función para configurar eventos táctiles
function setupTouchEvents() {
    const galleryContainer = document.querySelector('.gallery-container');
    
    galleryContainer.addEventListener('touchstart', (e) => {
        touchStartX = e.touches[0].clientX;
    }, false);
    
    galleryContainer.addEventListener('touchend', (e) => {
        touchEndX = e.changedTouches[0].clientX;
        handleSwipe();
    }, false);
}

// Función para manejar el swipe
function handleSwipe() {
    const swipeThreshold = 50;
    const swipeDistance = touchEndX - touchStartX;
    
    if (Math.abs(swipeDistance) > swipeThreshold) {
        if (swipeDistance > 0) {
            // Swipe derecha - foto anterior
            navigateGallery('prev');
        } else {
            // Swipe izquierda - siguiente foto
            navigateGallery('next');
        }
    }
}

// Función para navegar por la galería
function navigateGallery(direction) {
    if (direction === 'prev') {
        currentPhotoIndex = (currentPhotoIndex - 1 + galleryPhotos.length) % galleryPhotos.length;
    } else {
        currentPhotoIndex = (currentPhotoIndex + 1) % galleryPhotos.length;
    }
    updateGalleryView();
}

// Event listeners para los botones de navegación
document.addEventListener('DOMContentLoaded', () => {
    const prevButton = document.querySelector('.prev-photo');
    const nextButton = document.querySelector('.next-photo');
    const galleryModal = document.getElementById('gallery-modal');
    const closeButton = galleryModal.querySelector('.modal-close');
    
    prevButton.addEventListener('click', () => navigateGallery('prev'));
    nextButton.addEventListener('click', () => navigateGallery('next'));
    
    // Cerrar el modal de galería
    closeButton.addEventListener('click', () => {
        galleryModal.style.display = 'none';
        document.body.style.overflow = 'auto';
    });
    
    // Cerrar al hacer clic fuera del contenido
    galleryModal.addEventListener('click', (e) => {
        if (e.target === galleryModal) {
            galleryModal.style.display = 'none';
            document.body.style.overflow = 'auto';
        }
    });
    
    // Navegación con teclado
    document.addEventListener('keydown', (e) => {
        if (galleryModal.style.display === 'block') {
            switch(e.key) {
                case 'ArrowLeft':
                    navigateGallery('prev');
                    break;
                case 'ArrowRight':
                    navigateGallery('next');
                    break;
                case 'Escape':
                    galleryModal.style.display = 'none';
                    document.body.style.overflow = 'auto';
                    break;
            }
        }
    });
}); 