// Función para cargar los datos del archivo JSON
async function loadResultsData(filename = 'data_10k.json') {
    try {
        // Añadir parámetro de timestamp para evitar el caché del navegador
        const cacheBuster = `?t=${Date.now()}`;
        const response = await fetch(`../data/${filename}${cacheBuster}`);
        
        // Verificar que la respuesta fue exitosa
        if (!response.ok) {
            console.error(`Error al cargar ${filename}: ${response.status} ${response.statusText}`);
            return [];
        }
        
        const data = await response.json();
        
        // Validar que data es un array
        if (!Array.isArray(data)) {
            console.error(`Error: ${filename} no contiene un array válido`);
            return [];
        }
        
        // Registrar información completa del primer corredor para depuración
        if (data.length > 0) {
            console.log('Datos JSON cargados para', filename);
            console.log('Primer corredor:', data[0]);
            console.log('¿Tiene UniqueId?', data[0].hasOwnProperty('UniqueId'));
            if (data[0].UniqueId) {
                console.log('Valor de UniqueId:', data[0].UniqueId);
            } else {
                console.error('⚠️ ADVERTENCIA: No se encontró UniqueId en los datos JSON');
            }
        } else {
            console.warn('No se encontraron datos en el archivo JSON');
            return [];
        }
        
        // Transformar los datos al formato esperado por la aplicación
        const mappedData = data.map(runner => {
            // Determinar la distancia según el nombre del archivo
            let distanceValue = '10k';
            let distanceNum = 10;
            
            if (filename.includes('50k')) {
                distanceValue = '50k';
                distanceNum = 50;
            } else if (filename.includes('42k')) {
                distanceValue = '42k';
                distanceNum = 42.195;
            } else if (filename.includes('25k')) {
                distanceValue = '25k';
                distanceNum = 25;
            } else if (filename.includes('21k')) {
                distanceValue = '21k';
                distanceNum = 21.0975;
            } else if (filename.includes('15k')) {
                distanceValue = '15k';
                distanceNum = 15;
            } else if (filename.includes('10k')) {
                distanceValue = '10k';
                distanceNum = 10;
            } else if (filename.includes('5k')) {
                distanceValue = '5k';
                distanceNum = 5;
            } else if (filename.includes('3k')) {
                distanceValue = '3k';
                distanceNum = 3;
            }
            
            // Detectar parciales dinámicamente (campos tipo "5.0 km", "10 km", "MP1", etc)
            const parciales = {};
            for (const key in runner) {
                // Buscar campos con formato "X.X km" o "XX km" o "MPX"
                if (key.match(/^\d+(\.\d+)?\s*km$/i) || key.match(/^MP\d+$/i) || key.match(/^Parcial\s+\d+$/i)) {
                    parciales[key] = runner[key];
                }
            }
            
            return {
                position: runner.Pos,
                name: runner.Athlete || runner.Name,
                bib: (runner.Bib || runner["Bib#"] || runner.bib || '').toString(),
                category: runner.Category || runner.Class || runner.category,
                gender: runner.Gender || runner.gender,
                distance: distanceValue,
                parciales: parciales,
                finishTime: {
                    time: runner.FinishTime || runner['Chip Time'] || runner.finishTime,
                    pace: calculatePace(runner.FinishTime || runner['Chip Time'] || runner.finishTime, distanceNum)
                },
                gunTime: {
                    time: runner.GunTime || runner['Gun Time'] || runner.gunTime,
                    pace: calculatePace(runner.GunTime || runner['Gun Time'] || runner.gunTime, distanceNum)
                },
                UniqueId: runner.UniqueId
            };
        });
        
        // Verificar que los datos mapeados incluyen UniqueId
        if (mappedData.length > 0) {
            console.log('Datos mapeados con UniqueId:', mappedData[0].UniqueId);
        }
        
        return mappedData;
    } catch (error) {
        console.error('Error al cargar los datos:', error);
        return [];
    }
}

// Función para calcular el ritmo (pace) en km/h
function calculatePace(time, distance) {
    if (!time) return "0 km/h";
    
    const [hours, minutes, seconds] = time.split(':').map(Number);
    const totalSeconds = hours * 3600 + minutes * 60 + seconds;
    const pace = (distance / totalSeconds) * 3600; // km/h
    
    return pace.toFixed(2) + " km/h";
}

// Variables globales para almacenar los datos
let resultsData = [];
let currentFilteredResults = [];

// Función para recargar los datos según la distancia
window.reloadResultsData = async function(distance) {
    let files = [];
    if (!distance) {
        files = ['data_3k.json', 'data_5k.json', 'data_10k.json'];
    } else if (distance === '3k') {
        files = ['data_3k.json'];
    } else if (distance === '5k') {
        files = ['data_5k.json'];
    } else {
        files = ['data_10k.json'];
    }
    let allData = [];
    for (const file of files) {
        const data = await loadResultsData(file);
        allData = allData.concat(data);
    }
    resultsData = allData;
    currentFilteredResults = [...allData];
    document.dispatchEvent(new CustomEvent('resultsDataLoaded'));
}

// Cargar los datos cuando se inicie la aplicación (por defecto 10k)
document.addEventListener('DOMContentLoaded', function() {
    loadResultsData('data_10k.json').then(data => {
        resultsData = data;
        currentFilteredResults = [...data];
        // Disparar un evento personalizado cuando los datos estén listos
        document.dispatchEvent(new CustomEvent('resultsDataLoaded'));
    });
});