// Función para generar código QR
function generateQRCode(uniqueId) {
    // Verificar que tenemos un ID único válido
    if (!uniqueId) return;

    // Crear la URL absoluta completa para el certificado (solo con el ID)
    let baseUrl = window.location.href.split('?')[0]; // Elimina cualquier parámetro existente
    if (!baseUrl.startsWith('http')) {
        // Si no tenemos el protocolo, construimos la URL completa
        baseUrl = `${window.location.protocol}//${window.location.host}${window.location.pathname}`;
    }
    
    const certificateUrl = `${baseUrl}?id=${encodeURIComponent(uniqueId)}`;
    
    console.log('URL del QR generada:', certificateUrl);
    
    // Determinar la versión adecuada del QR según la longitud de la URL
    let qrVersion = 0; // Auto-detectar
    if (certificateUrl.length > 114) {
        qrVersion = 8; // Versión más grande para URLs más largas
    }
    
    // Generar código QR con la URL
    const qr = qrcode(qrVersion, 'M');
    qr.addData(certificateUrl);
    qr.make();
    
    // Mostrar el QR en la imagen
    const qrImg = document.getElementById('runner-qr');
    if (qrImg) {
        // Intentar reemplazar con el QR generado
        try {
            const qrSize = 10; // Tamaño de los módulos QR en píxeles
            const qrMargin = 4; // Margen alrededor del QR
            qrImg.src = qr.createDataURL(qrSize, qrMargin);
            qrImg.alt = "Escanea para verificar este certificado";
            qrImg.title = certificateUrl;
        } catch (e) {
            console.error('Error al generar el código QR:', e);
        }
    }
}

// Función para cargar datos del certificado desde los archivos JSON
async function loadRunnerDataFromJSON(uniqueId) {
    if (!uniqueId) return false;

    const distances = ['3k', '5k', '10k'];
    let errorCount = 0;
    
    // PASO 1: Intentar cargar desde archivos JSON directamente (método del cliente)
    for (const distance of distances) {
        try {
            console.log(`Buscando ID ${uniqueId} en data_${distance}.json...`);
            const response = await fetch(`data/data_${distance}.json?cache=${Date.now()}`); // Evitar caché
            
            if (!response.ok) {
                console.warn(`No se pudo cargar data_${distance}.json, código: ${response.status}`);
                errorCount++;
                continue;
            }
            
            const runners = await response.json();
            console.log(`Datos cargados para ${distance}: ${runners.length} corredores`);
            
            const runner = runners.find(r => r.UniqueId === uniqueId);
            
            if (runner) {
                console.log(`Corredor encontrado en ${distance}:`, runner);
                // Encontramos el corredor, actualizar el certificado
                updateCertificateWithRunnerData(runner, distance);
                return true;
            }
        } catch (e) {
            console.error(`Error al cargar datos para ${distance}:`, e);
            errorCount++;
        }
    }
    
    // PASO 2: Si falló la carga directa, intentar usando el script PHP como respaldo
    try {
        console.log('Intentando buscar por PHP como fallback...');
        const phpResponse = await fetch(`php/get-runner-by-id.php?id=${encodeURIComponent(uniqueId)}`);
        const result = await phpResponse.json();
        
        if (result.success && result.data) {
            console.log('Corredor encontrado a través de PHP:', result.data);
            // Encontramos el corredor, actualizar el certificado
            updateCertificateWithRunnerData(result.data, result.source);
            return true;
        } else {
            console.warn('No se encontró el corredor a través de PHP:', result.message);
        }
    } catch (e) {
        console.error('Error al buscar por PHP:', e);
    }
    
    // Si todas las cargas fallaron, registrar un error especial
    if (errorCount === distances.length) {
        console.error('Error crítico: No se pudo cargar ningún archivo de datos.');
    }
    
    return false;
}

// Actualizar el certificado con los datos obtenidos del JSON
function updateCertificateWithRunnerData(runner, distance) {
    // Actualizar los elementos del DOM con los datos del corredor
    const elements = {
        'runner-name': runner.Name,
        'runner-category': runner.Class,
        'runner-distance': distance,
        'runner-position': `${runner.Pos}°`,
        'runner-chip-time': runner.FinishTime || '--:--:--',
        'runner-official-time': runner.GunTime || '--:--:--',
    };
    
    // Actualizar cada elemento
    for (const [id, value] of Object.entries(elements)) {
        const element = document.getElementById(id);
        if (element && value) element.textContent = value;
    }
    
    // Actualizar el ID de verificación
    const runnerBibElement = document.getElementById('runner-bib');
    if (runnerBibElement) {
        runnerBibElement.textContent = runner.Bib || '---';
    }
    
    // Para compatibilidad con elementos antiguos
    document.querySelectorAll('.verification-id:not(:has(#runner-bib))').forEach(el => {
        el.textContent = `ID: ${runner.Bib || '---'}`;
    });
    
    // Mostrar el ID único en todos los lugares correspondientes
    document.getElementById('unique-id-display').textContent = runner.UniqueId;
    document.getElementById('header-unique-id').textContent = runner.UniqueId;
    
    // Calcular el ritmo si tenemos tiempo y distancia
    if (runner.FinishTime && distance) {
        calculatePace(runner.FinishTime, distance);
    }
    
    // Generar el código QR con el ID único
    generateQRCode(runner.UniqueId);
}

// Calcular el ritmo por kilómetro
function calculatePace(chipTime, distance) {
    // Obtener la distancia numérica
    let distanceNum = parseFloat(distance);
    if (isNaN(distanceNum)) {
        // Intentar extraer el número si viene como '10K' o '10k'
        const match = distance.match(/(\d+(?:\.\d+)?)/);
        if (match) distanceNum = parseFloat(match[1]);
    }
    
    if (distanceNum > 0) {
        const [h, m, s] = chipTime.split(':').map(Number);
        const totalSeconds = h * 3600 + m * 60 + s;
        const paceSeconds = totalSeconds / distanceNum;
        const paceMin = Math.floor(paceSeconds / 60);
        const paceSec = Math.round(paceSeconds % 60).toString().padStart(2, '0');
        document.getElementById('runner-pace').textContent = `${paceMin}:${paceSec} min/km`;
    }
}

// Función para mostrar un mensaje de error en el certificado
function showCertificateError(message) {
    // 1. Actualizar título superior para mantener la cabecera roja en "Reconocimiento Oficial"
    const titleElement = document.querySelector('.certificate-subtitle');
    if (titleElement) {
        titleElement.textContent = "RECONOCIMIENTO OFICIAL";
    }
    
    // 2. Limpiar el contenido del cuerpo del certificado
    const certificateBody = document.querySelector('.certificate-body');
    if (certificateBody) {
        // Guardar el contenido original
        if (!certificateBody.dataset.originalContent) {
            certificateBody.dataset.originalContent = certificateBody.innerHTML;
        }
        
        // Mostrar mensaje de error con formato similar al diseño del certificado
        certificateBody.innerHTML = `
            <div class="certificate-error" style="text-align: center; padding: 60px 20px;">
                <h1 style="color: #e74c3c; margin-bottom: 30px; font-size: 28px;">Error al cargar el certificado</h1>
                <p style="font-size: 20px; margin: 30px 0; color: #333;">${message}</p>
                <p style="font-size: 16px; color: #666;">Por favor, verifica el ID proporcionado o contacta al administrador.</p>
            </div>
        `;
    }
    
    // 3. Ocultar los botones de acción y redes sociales
    const footerButtons = document.querySelector('.footer-buttons');
    if (footerButtons) {
        footerButtons.style.display = 'none';
    }
    
    // 4. Asegurarnos de que el ID único muestra el valor proporcionado
    const uniqueIdDisplay = document.getElementById('unique-id-display');
    const headerUniqueId = document.getElementById('header-unique-id');
    const urlParams = new URLSearchParams(window.location.search);
    const uniqueId = urlParams.get('id') || '---';
    
    if (uniqueIdDisplay) {
        uniqueIdDisplay.textContent = uniqueId;
    }
    
    if (headerUniqueId) {
        headerUniqueId.textContent = uniqueId;
    }
}

// Función para verificar la conexión a Internet
function checkInternetConnection() {
    return navigator.onLine;
}

// Inicializar al cargar el documento
document.addEventListener('DOMContentLoaded', async function() {
    // Verificar conexión a Internet
    if (!checkInternetConnection()) {
        showCertificateError('No se pudo conectar al servidor. Verifica tu conexión a Internet.');
        console.error('No hay conexión a Internet disponible');
        return;
    }
    
    // Obtener los parámetros de la URL
    const urlParams = new URLSearchParams(window.location.search);
    
    // Si tenemos un ID en la URL, cargar datos desde JSON
    const uniqueId = urlParams.get('id');
    if (uniqueId) {
        console.log('Cargando datos para el ID único:', uniqueId);
        
        try {
            // Mostrar mensaje de carga
            const certificateBody = document.querySelector('.certificate-body');
            if (certificateBody) {
                certificateBody.innerHTML += `
                    <div id="loading-indicator" style="position: fixed; top: 50%; left: 50%; transform: translate(-50%, -50%); 
                        background: rgba(255,255,255,0.9); padding: 20px; border-radius: 10px; 
                        box-shadow: 0 0 10px rgba(0,0,0,0.2); text-align: center; z-index: 9999;">
                        <div style="margin-bottom: 15px;">
                            <i class="fas fa-spinner fa-spin" style="font-size: 30px; color: #3498db;"></i>
                        </div>
                        <p style="margin: 0; font-size: 16px;">Cargando certificado...</p>
                    </div>
                `;
            }
            
            const loaded = await loadRunnerDataFromJSON(uniqueId);
            
            // Remover indicador de carga
            const loadingIndicator = document.getElementById('loading-indicator');
            if (loadingIndicator) loadingIndicator.remove();
            
            if (!loaded) {
                // Si no se encontraron datos, mostrar mensaje de error
                showCertificateError(`No se encontraron datos para el ID: ${uniqueId}`);
            }
        } catch (error) {
            console.error('Error al cargar datos del corredor:', error);
            showCertificateError('Error al procesar los datos del certificado.');
        }
    } else {
        // Si no hay ID pero hay otros parámetros, usar el enfoque anterior (compatibilidad)
        const oldParams = ['name', 'category', 'distance', 'position', 'chipTime', 'officialTime', 'bib'];
        const hasOldParams = oldParams.some(param => urlParams.has(param));
        
        if (hasOldParams) {
            // Estos casos serán manejados por el script integration.js original
            console.log('Usando el modo de compatibilidad (parámetros antiguos en URL)');
            
            const id = urlParams.get('id'); // Podría ser null
            if (id) {
                generateQRCode(id);
            }
        } else {
            // No hay parámetros en absoluto
            console.log('No se proporcionaron parámetros para el certificado');
            showCertificateError('No se proporcionó ningún ID para el certificado.');
        }
    }
});
