// Variables globales
let selectedRunners = [];
let performanceChart = null;

// Colores para las gráficas
const chartColors = [
    '#e63946', // Rojo
    '#457b9d', // Azul
    '#2a9d8f', // Verde
    '#e9c46a', // Amarillo
    '#f4a261', // Naranja
    '#264653', // Azul oscuro
    '#8338ec', // Púrpura
    '#ff006e'  // Rosa
];

// Inicialización
document.addEventListener('DOMContentLoaded', function() {
    initializeSearch();
    initializeAddButton();
    setupDownloadButtons();
});

// Inicializar la búsqueda
function initializeSearch() {
    const searchInput = document.getElementById('runner-search');
    searchInput.addEventListener('input', debounce(async function(e) {
        const query = e.target.value.trim();
        if (query.length < 2) return;

        const results = await searchRunners(query);
        showSearchResults(results);
    }, 300));
}

// Inicializar el botón de agregar
function initializeAddButton() {
    const addButton = document.getElementById('add-runner-btn');
    addButton.addEventListener('click', async function() {
        const searchInput = document.getElementById('runner-search');
        const query = searchInput.value.trim();
        
        if (query.length < 2) {
            alert('Por favor, ingresa al menos 2 caracteres para buscar');
            return;
        }

        const results = await searchRunners(query);
        if (results.length > 0) {
            showRunnerSelectionDialog(results);
        } else {
            alert('No se encontraron corredores con ese criterio de búsqueda');
        }
    });
}

// Buscar corredores
async function searchRunners(query) {
    try {
        const allData = await Promise.all([
            loadResultsData('data_3k.json'),
            loadResultsData('data_5k.json'),
            loadResultsData('data_10k.json')
        ]);

        const allRunners = allData.flat();
        
        return allRunners.filter(runner => {
            const searchStr = `${runner.name} ${runner.bib}`.toLowerCase();
            return searchStr.includes(query.toLowerCase());
        });
    } catch (error) {
        console.error('Error al buscar corredores:', error);
        return [];
    }
}

// Mostrar resultados de búsqueda
function showSearchResults(results) {
    // Eliminar dropdown anterior si existe
    const existingDropdown = document.querySelector('.search-results-dropdown');
    if (existingDropdown) {
        existingDropdown.remove();
    }

    if (results.length === 0) return;

    const searchInput = document.getElementById('runner-search');
    const dropdown = document.createElement('div');
    dropdown.className = 'search-results-dropdown';

    results.forEach(runner => {
        const item = document.createElement('div');
        item.className = 'search-result-item';
        item.innerHTML = `
            <div class="name">${runner.name}</div>
            <div class="details">
                Dorsal: ${runner.bib} | Categoría: ${runner.category} | Distancia: ${runner.distance}
            </div>
        `;

        item.addEventListener('click', () => {
            addRunnerToComparison(runner);
            dropdown.remove();
            searchInput.value = '';
        });

        dropdown.appendChild(item);
    });

    // Posicionar el dropdown debajo del input
    const searchBox = document.querySelector('.search-box');
    searchBox.style.position = 'relative';
    searchBox.appendChild(dropdown);

    // Cerrar el dropdown al hacer clic fuera
    document.addEventListener('click', function closeDropdown(e) {
        if (!searchBox.contains(e.target)) {
            dropdown.remove();
            document.removeEventListener('click', closeDropdown);
        }
    });
}

// Mostrar diálogo de selección de corredor
function showRunnerSelectionDialog(results) {
    // Crear el modal
    const modal = document.createElement('div');
    modal.className = 'modal';
    modal.style.display = 'block';

    // Crear el contenido del modal
    const modalContent = document.createElement('div');
    modalContent.className = 'modal-content';
    modalContent.style.maxWidth = '600px';

    // Crear el encabezado
    const header = document.createElement('div');
    header.className = 'modal-header';
    header.innerHTML = `
        <h2>Seleccionar Corredor</h2>
        <span class="modal-close">&times;</span>
    `;

    // Crear la lista de resultados
    const resultsList = document.createElement('div');
    resultsList.className = 'modal-body';
    resultsList.style.maxHeight = '400px';
    resultsList.style.overflowY = 'auto';

    results.forEach(runner => {
        const item = document.createElement('div');
        item.className = 'runner-selection-item';
        item.innerHTML = `
            <div class="runner-info">
                <div class="name">${runner.name}</div>
                <div class="details">
                    Dorsal: ${runner.bib} | Categoría: ${runner.category} | Distancia: ${runner.distance}
                </div>
            </div>
            <button class="select-runner-btn">Seleccionar</button>
        `;

        item.querySelector('.select-runner-btn').addEventListener('click', () => {
            addRunnerToComparison(runner);
            modal.remove();
        });

        resultsList.appendChild(item);
    });

    // Ensamblar el modal
    modalContent.appendChild(header);
    modalContent.appendChild(resultsList);
    modal.appendChild(modalContent);
    document.body.appendChild(modal);

    // Cerrar el modal
    const closeBtn = modal.querySelector('.modal-close');
    closeBtn.addEventListener('click', () => modal.remove());

    // Cerrar al hacer clic fuera del modal
    modal.addEventListener('click', (e) => {
        if (e.target === modal) {
            modal.remove();
        }
    });
}

// Agregar corredor a la comparación
function addRunnerToComparison(runner) {
    if (selectedRunners.length >= 8) {
        alert('Solo puedes comparar hasta 8 corredores a la vez');
        return;
    }

    if (selectedRunners.some(r => r.bib === runner.bib)) {
        alert('Este corredor ya está en la comparación');
        return;
    }

    selectedRunners.push(runner);
    updateSelectedRunnersList();
    updatePerformanceChart();
}

// Actualizar lista de corredores seleccionados
function updateSelectedRunnersList() {
    const container = document.getElementById('selected-runners-list');
    container.innerHTML = '';

    selectedRunners.forEach((runner, index) => {
        const runnerCard = document.createElement('div');
        runnerCard.className = 'runner-card';
        runnerCard.innerHTML = `
            <div class="runner-info">
                <span class="runner-name">${runner.name}</span>
                <span class="runner-bib">Dorsal: ${runner.bib}</span>
                <span class="runner-category">${runner.category}</span>
            </div>
            <button class="remove-runner" data-index="${index}">
                <i class="fas fa-times"></i>
            </button>
        `;

        runnerCard.querySelector('.remove-runner').addEventListener('click', () => {
            selectedRunners.splice(index, 1);
            updateSelectedRunnersList();
            updatePerformanceChart();
            updateMetricsTable();
        });

        container.appendChild(runnerCard);
    });
    // Actualizar la tabla de métricas cada vez que se actualiza la lista
    updateMetricsTable();
    // Mostrar/ocultar botones de descarga
    toggleDownloadButtons();
}

// Actualizar gráfica de rendimiento
function updatePerformanceChart() {
    const ctx = document.getElementById('performanceComparisonChart').getContext('2d');
    
    if (performanceChart) {
        performanceChart.destroy();
    }

    // Construir etiquetas dinámicas basadas en los parciales disponibles
    const labels = [];
    const parcialesKeys = [];
    
    if (selectedRunners.length > 0) {
        // Obtener todas las claves de parciales del primer corredor
        const firstRunner = selectedRunners[0];
        if (firstRunner.parciales) {
            parcialesKeys.push(...Object.keys(firstRunner.parciales).sort((a, b) => {
                // Ordenar por distancia si tienen formato numérico
                const distA = parseFloat(a.match(/(\d+(?:\.\d+)?)/)?.[1] || 0);
                const distB = parseFloat(b.match(/(\d+(?:\.\d+)?)/)?.[1] || 0);
                return distA - distB;
            }));
            
            // Crear etiquetas legibles
            parcialesKeys.forEach(key => {
                labels.push(key.replace(/\s*km$/i, ' km'));
            });
        }
        labels.push('Tiempo Chip', 'Tiempo Oficial');
    }

    const datasets = selectedRunners.map((runner, index) => {
        const data = [];
        
        // Agregar tiempos de parciales
        parcialesKeys.forEach(key => {
            const parcialeTime = runner.parciales?.[key];
            data.push(timeToMinutes(parcialeTime || '00:00:00'));
        });
        
        // Agregar tiempos finales
        data.push(timeToMinutes(runner.finishTime?.time || '00:00:00'));
        data.push(timeToMinutes(runner.gunTime?.time || '00:00:00'));
        
        return {
            label: `${runner.name} (${runner.bib})`,
            data: data,
            borderColor: chartColors[index % chartColors.length],
            backgroundColor: `${chartColors[index % chartColors.length]}33`,
            fill: false,
            tension: 0.3,
            pointRadius: 5,
            pointBackgroundColor: chartColors[index % chartColors.length],
            pointBorderColor: '#1d3557',
            pointHoverRadius: 7
        };
    });

    performanceChart = new Chart(ctx, {
        type: 'line',
        data: {
            labels: labels,
            datasets: datasets
        },
        options: {
            responsive: true,
            plugins: {
                legend: {
                    display: true,
                    position: 'top'
                },
                title: {
                    display: true,
                    text: 'Comparativa de Rendimiento (minutos)'
                }
            },
            scales: {
                y: {
                    title: {
                        display: true,
                        text: 'Minutos'
                    },
                    beginAtZero: true
                }
            }
        }
    });
}

// Convertir tiempo a minutos
function timeToMinutes(time) {
    if (!time) return 0;
    const parts = time.split(':').map(Number);
    if (parts.length === 3) return parts[0]*60 + parts[1] + parts[2]/60;
    if (parts.length === 2) return parts[0]*60 + parts[1];
    return Number(time);
}

// Función debounce para optimizar búsquedas
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Agregar estilos para el modal
const style = document.createElement('style');
style.textContent = `
    .modal {
        display: none;
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0, 0, 0, 0.5);
        z-index: 1000;
    }

    .modal-content {
        background-color: #fff;
        margin: 10% auto;
        padding: 20px;
        border-radius: 10px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    }

    .modal-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 20px;
        padding-bottom: 10px;
        border-bottom: 1px solid #dee2e6;
    }

    .modal-close {
        font-size: 24px;
        cursor: pointer;
        color: #666;
    }

    .modal-close:hover {
        color: #e63946;
    }

    .runner-selection-item {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 15px;
        border-bottom: 1px solid #eee;
    }

    .runner-selection-item:last-child {
        border-bottom: none;
    }

    .select-runner-btn {
        background-color: #457b9d;
        color: white;
        border: none;
        padding: 8px 16px;
        border-radius: 5px;
        cursor: pointer;
        transition: background-color 0.3s;
    }

    .select-runner-btn:hover {
        background-color: #1d3557;
    }
`;
document.head.appendChild(style);

// Calcular ritmo promedio (min/km)
function calcularRitmoMinKm(tiempo, distancia) {
    if (!tiempo || !distancia) return '--:--';
    const partes = tiempo.split(':').map(Number);
    let segundos = 0;
    if (partes.length === 3) segundos = partes[0]*3600 + partes[1]*60 + partes[2];
    else if (partes.length === 2) segundos = partes[0]*60 + partes[1];
    else segundos = Number(tiempo);
    const ritmo = segundos / distancia;
    const min = Math.floor(ritmo / 60);
    const seg = Math.round(ritmo % 60).toString().padStart(2, '0');
    return `${min}:${seg}`;
}

// Calcular velocidad promedio (km/h)
function calcularVelocidadKmH(tiempo, distancia) {
    if (!tiempo || !distancia) return '--';
    const partes = tiempo.split(':').map(Number);
    let segundos = 0;
    if (partes.length === 3) segundos = partes[0]*3600 + partes[1]*60 + partes[2];
    else if (partes.length === 2) segundos = partes[0]*60 + partes[1];
    else segundos = Number(tiempo);
    const horas = segundos / 3600;
    if (horas === 0) return '--';
    return (distancia / horas).toFixed(2);
}

// Actualizar tabla de métricas comparativas
function updateMetricsTable() {
    const metricsGrid = document.querySelector('.metrics-grid');
    if (!metricsGrid) return;
    if (selectedRunners.length === 0) {
        metricsGrid.innerHTML = '<p style="text-align:center;color:#888;">Agrega corredores para comparar sus métricas.</p>';
        return;
    }

    // Calcular mejores valores para resaltar
    let mejorTiempoChip = Math.min(...selectedRunners.map(r => timeToMinutes(r.finishTime.time)));
    let mejorRitmo = Math.min(...selectedRunners.map(r => {
        const dist = parseFloat(r.distance.replace('k','').replace('K',''));
        return timeToMinutes(r.finishTime.time) / (dist || 1);
    }));
    let mejorVelocidad = Math.max(...selectedRunners.map(r => {
        const dist = parseFloat(r.distance.replace('k','').replace('K',''));
        return Number(calcularVelocidadKmH(r.finishTime.time, dist));
    }));

    // Construir la tabla responsiva SIN ICONOS y SIN 'Pos. Cat.'
    let html = `<div style=\"overflow-x:auto;\"><table class=\"metrics-table\" style=\"width:100%;border-collapse:collapse;text-align:center;\">
        <thead>
            <tr style=\"background:#f8f9fa;\">
                <th>Nombre</th>
                <th>Dorsal</th>
                <th>Distancia</th>
                <th>Categoría</th>
                <th>Tiempo Chip</th>
                <th>Tiempo Oficial</th>
                <th>Ritmo (min/km)</th>
                <th>Velocidad (km/h)</th>
                <th>Pos. Gral</th>
            </tr>
        </thead>
        <tbody>`;
    selectedRunners.forEach(runner => {
        const dist = parseFloat(runner.distance.replace('k','').replace('K',''));
        const ritmo = calcularRitmoMinKm(runner.finishTime.time, dist);
        const velocidad = calcularVelocidadKmH(runner.finishTime.time, dist);
        html += `<tr>
            <td data-label=\"Nombre\">${runner.name}</td>
            <td data-label=\"Dorsal\">${runner.bib}</td>
            <td data-label=\"Distancia\">${runner.distance}</td>
            <td data-label=\"Categoría\">${runner.category}</td>
            <td data-label=\"Tiempo Chip\" data-best=\"${timeToMinutes(runner.finishTime.time) === mejorTiempoChip}\">${runner.finishTime.time}</td>
            <td data-label=\"Tiempo Oficial\">${runner.gunTime.time}</td>
            <td data-label=\"Ritmo (min/km)\" data-best=\"${(timeToMinutes(runner.finishTime.time) / (dist || 1)) === mejorRitmo}\">${ritmo}</td>
            <td data-label=\"Velocidad (km/h)\" data-best=\"${Number(velocidad) === mejorVelocidad}\">${velocidad}</td>
            <td data-label=\"Pos. Gral\">${runner.position || '--'}</td>
        </tr>`;
    });
    html += '</tbody></table></div>';
    metricsGrid.innerHTML = html;
}

// Mostrar u ocultar los botones de descarga según si hay resultados
function toggleDownloadButtons() {
    const btns = document.getElementById('download-buttons');
    if (!btns) return;
    if (selectedRunners.length > 0) {
        btns.style.display = 'flex';
    } else {
        btns.style.display = 'none';
    }
}

function hideNoExportElements() {
    document.querySelectorAll('.no-export').forEach(el => el.style.display = 'none');
}
function showNoExportElements() {
    document.querySelectorAll('.no-export').forEach(el => el.style.display = '');
}

// Descargar gráfica + tabla + portada como PNG
function downloadChartAsPNG() {
    const area = document.getElementById('export-area');
    if (!area) return;
    hideNoExportElements();
    html2canvas(area).then(canvas => {
        showNoExportElements();
        const link = document.createElement('a');
        link.href = canvas.toDataURL('image/png');
        link.download = 'comparativa_rendimiento.png';
        link.click();
    });
}

// Descargar gráfica + tabla + portada como PDF
function downloadChartAsPDF() {
    const area = document.getElementById('export-area');
    if (!area) return;
    hideNoExportElements();
    html2canvas(area).then(canvas => {
        showNoExportElements();
        const imgData = canvas.toDataURL('image/png');
        const pdf = new window.jspdf.jsPDF({orientation: 'landscape', unit: 'pt', format: 'a4'});
        const pageWidth = pdf.internal.pageSize.getWidth();
        const imgWidth = pageWidth - 60;
        const imgHeight = canvas.height * (imgWidth / canvas.width);
        pdf.addImage(imgData, 'PNG', 30, 40, imgWidth, imgHeight);
        pdf.save('comparativa_rendimiento.pdf');
    });
}

// Asignar eventos a los botones y configurar el modal de explicación
function setupDownloadButtons() {
    const btnPng = document.getElementById('btn-download-png');
    const btnPdf = document.getElementById('btn-download-pdf');
    const btnShowExplanation = document.getElementById('btn-show-explanation');
    const explanationModal = document.getElementById('explanation-modal');
    const closeModalBtn = explanationModal ? explanationModal.querySelector('.modal-close') : null;

    if (btnPng) btnPng.onclick = downloadChartAsPNG;
    if (btnPdf) btnPdf.onclick = downloadChartAsPDF;

    // Mostrar el modal de explicación al hacer clic en el botón
    if (btnShowExplanation && explanationModal) {
        btnShowExplanation.addEventListener('click', () => {
            explanationModal.style.display = 'block';
        });
    }

    // Ocultar el modal al hacer clic en la X
    if (closeModalBtn && explanationModal) {
        closeModalBtn.addEventListener('click', () => {
            explanationModal.style.display = 'none';
        });
    }

    // Ocultar el modal al hacer clic fuera de él
    if (explanationModal) {
        window.addEventListener('click', (event) => {
            if (event.target === explanationModal) {
                explanationModal.style.display = 'none';
            }
        });
    }
} 